from utils import Gtk, Adw, EINHEITEN, partial, show_message, confirm
from datetime import datetime

def create_einkauf_page(window):
    page = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=20)
    page.set_margin_start(40); page.set_margin_end(40)
    page.set_margin_top(40); page.set_margin_bottom(40)
    header = Gtk.Label(label="Einkauf erfassen")
    header.add_css_class('title-2'); header.set_halign(Gtk.Align.START)
    page.append(header)
    form = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=15)

    def create_form_row(label_text, widget):
        box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
        label = Gtk.Label(label=label_text)
        label.set_size_request(150, -1); label.set_halign(Gtk.Align.START)
        box.append(label)
        if isinstance(widget, list):
            for w in widget:
                box.append(w)
        else:
            box.append(widget)
        form.append(box)

    window.einkauf_datum = Gtk.Entry(); window.einkauf_datum.set_text(datetime.now().strftime('%Y-%m-%d'))
    create_form_row("Datum:", window.einkauf_datum)

    window.einkauf_rechnungs_nr = Gtk.Entry()
    create_form_row("Rechnungs-Nr.:", window.einkauf_rechnungs_nr)

    window.einkauf_artikel_dd = Gtk.DropDown()
    window.einkauf_artikel_dd.set_enable_search(True)
    create_form_row("Artikel:", window.einkauf_artikel_dd)

    window.einkauf_menge = Gtk.SpinButton(); window.einkauf_menge.set_adjustment(Gtk.Adjustment(value=1, lower=0.0001, upper=1000000, step_increment=1))
    window.einkauf_einheit = Gtk.DropDown.new_from_strings(EINHEITEN); window.einkauf_einheit.set_selected(0)
    create_form_row("Menge:", [window.einkauf_menge, window.einkauf_einheit])

    window.einkauf_steuer = Gtk.DropDown.new_from_strings(["7%", "19%"]); window.einkauf_steuer.set_selected(1)
    create_form_row("Steuersatz:", window.einkauf_steuer)

    preis_box_widgets = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=5)
    window.einkauf_preis_netto = Gtk.Entry(); window.einkauf_preis_netto.set_text("0,00")
    window.einkauf_preis_netto.set_hexpand(True)
    window.einkauf_preis_brutto = Gtk.Entry(); window.einkauf_preis_brutto.set_text("0,00")
    window.einkauf_preis_brutto.set_hexpand(True)
    preis_box_widgets.append(window.einkauf_preis_netto)
    preis_box_widgets.append(Gtk.Label(label="€ Netto /"))
    preis_box_widgets.append(window.einkauf_preis_brutto)
    preis_box_widgets.append(Gtk.Label(label="€ Brutto"))
    create_form_row("Preis pro Einheit:", preis_box_widgets)
    
    window.einkauf_lieferant_dd = Gtk.DropDown()
    window.einkauf_lieferant_dd.set_enable_search(True)
    create_form_row("Lieferant:", window.einkauf_lieferant_dd)
    page.append(form)

    window.einkauf_preis_netto.connect('changed', partial(on_netto_preis_changed, window))
    window.einkauf_preis_brutto.connect('changed', partial(on_brutto_preis_changed, window))
    window.einkauf_steuer.connect('notify::selected', partial(on_steuer_changed, window))

    btn = Gtk.Button(label="Einkauf speichern"); btn.add_css_class('suggested-action'); btn.set_size_request(200, -1)
    btn.connect('clicked', partial(on_einkauf_save, window))
    page.append(btn)
    window.content_stack.add_named(page, 'einkauf')

def on_steuer_changed(window, dropdown, param):
    on_netto_preis_changed(window, window.einkauf_preis_netto)

def on_netto_preis_changed(window, entry):
    if window._is_updating_prices: return
    window._is_updating_prices = True
    try:
        netto = float(entry.get_text().replace(',', '.'))
        steuer_idx = window.einkauf_steuer.get_selected()
        steuer_satz = 0.07 if steuer_idx == 0 else 0.19
        brutto = netto * (1 + steuer_satz)
        window.einkauf_preis_brutto.set_text(f"{brutto:.4f}".replace('.', ','))
    except ValueError:
        window.einkauf_preis_brutto.set_text("")
    finally:
        window._is_updating_prices = False

def on_brutto_preis_changed(window, entry):
    if window._is_updating_prices: return
    window._is_updating_prices = True
    try:
        brutto = float(entry.get_text().replace(',', '.'))
        steuer_idx = window.einkauf_steuer.get_selected()
        steuer_satz = 0.07 if steuer_idx == 0 else 0.19
        netto = brutto / (1 + steuer_satz)
        window.einkauf_preis_netto.set_text(f"{netto:.4f}".replace('.', ','))
    except ValueError:
        window.einkauf_preis_netto.set_text("")
    finally:
        window._is_updating_prices = False

def on_einkauf_save(window, button):
    try:
        datum = window.einkauf_datum.get_text()
        rechnungs_nr = window.einkauf_rechnungs_nr.get_text().strip()

        artikel_idx = window.einkauf_artikel_dd.get_selected()
        if artikel_idx == Gtk.INVALID_LIST_POSITION:
            show_message(window, "Fehler", "Bitte einen Artikel auswählen."); return
        artikel = window.einkauf_artikel_dd.get_model().get_string(artikel_idx)

        menge = float(window.einkauf_menge.get_value())
        preis_netto = float(window.einkauf_preis_netto.get_text().replace(',', '.'))
        preis_brutto = float(window.einkauf_preis_brutto.get_text().replace(',', '.'))
        steuer = 7 if window.einkauf_steuer.get_selected() == 0 else 19

        lieferant_idx = window.einkauf_lieferant_dd.get_selected()
        if lieferant_idx == Gtk.INVALID_LIST_POSITION:
            show_message(window, "Fehler", "Bitte einen Lieferanten auswählen."); return
        lieferant = window.einkauf_lieferant_dd.get_model().get_string(lieferant_idx)
        
        einheit = EINHEITEN[window.einkauf_einheit.get_selected()]

        einkauf = {
            'datum': datum, 'artikel': artikel, 'menge': menge, 'einheit': einheit,
            'preis_netto': preis_netto, 'preis_brutto': preis_brutto,
            'steuer': steuer, 'lieferant': lieferant,
            'gesamt_brutto': round(menge * preis_brutto, 2), 'rechnungs_nr': rechnungs_nr
        }
        window.app.data['einkäufe'].append(einkauf)

        bestand_item = next((b for b in window.app.data['bestand'] if b['artikel'] == artikel), None)
        if bestand_item:
            bestand_item['menge'] += menge
            bestand_item['preis'] = preis_netto
        else:
            window.app.data['bestand'].append({'artikel': artikel, 'menge': menge, 'preis': preis_netto, 'einheit': einheit})

        window.app.save_data()
        show_message(window, "Erfolg", "Einkauf gespeichert")
        window.einkauf_preis_netto.set_text("0,00"); window.einkauf_rechnungs_nr.set_text("")
        window.einkauf_menge.set_value(1)
        
        update_einkauf_jahr_filter(window)
        refresh_einkauf_list(window)
        from bestand import refresh_bestand_list
        refresh_bestand_list(window)
        from kassenbuch import refresh_kassenbuch
        refresh_kassenbuch(window)
    except (ValueError, TypeError):
        show_message(window, "Fehler", "Bitte gültige Zahlen für Menge und Preis eingeben.")

def create_einkauf_list_page(window):
    page = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=20)
    page.set_margin_start(40); page.set_margin_end(40)
    page.set_margin_top(40); page.set_margin_bottom(40)
    header = Gtk.Label(label="Einkäufe"); header.add_css_class('title-2'); header.set_halign(Gtk.Align.START)
    page.append(header)
    filters_container = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=10)
    filter_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
    filter_label = Gtk.Label(label="Suchen:")
    window.einkauf_filter = Gtk.Entry(); window.einkauf_filter.set_placeholder_text("Artikel, Lieferant, Re-Nr...")
    window.einkauf_filter.connect('changed', lambda e: refresh_einkauf_list(window))
    filter_box.append(filter_label); filter_box.append(window.einkauf_filter)
    filters_container.append(filter_box)

    date_filter_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
    date_filter_label = Gtk.Label(label="Datum:")
    window.einkauf_filter_jahr = Gtk.DropDown()
    window.einkauf_filter_jahr.connect('notify::selected', lambda d, p: refresh_einkauf_list(window))
    monate = ["Alle Monate"] + [f"{i:02d}" for i in range(1, 13)]
    window.einkauf_filter_monat = Gtk.DropDown.new_from_strings(monate)
    window.einkauf_filter_monat.connect('notify::selected', lambda d, p: refresh_einkauf_list(window))
    date_filter_box.append(date_filter_label)
    date_filter_box.append(window.einkauf_filter_jahr)
    date_filter_box.append(window.einkauf_filter_monat)
    filters_container.append(date_filter_box)
    page.append(filters_container)
    
    update_einkauf_jahr_filter(window)

    scrolled = Gtk.ScrolledWindow(); scrolled.set_vexpand(True); scrolled.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.AUTOMATIC)
    window.einkauf_list_box = Gtk.ListBox(); window.einkauf_list_box.add_css_class('boxed-list')
    scrolled.set_child(window.einkauf_list_box)
    page.append(scrolled)
    window.content_stack.add_named(page, 'einkauf_list')

def update_einkauf_jahr_filter(window):
    jahre = {"Alle Jahre"}
    for einkauf in window.app.data['einkäufe']:
        try:
            jahre.add(einkauf['datum'][:4])
        except (IndexError, TypeError):
            continue
    
    jahre_liste = sorted(list(jahre), reverse=True)
    if "Alle Jahre" in jahre_liste:
         jahre_liste.remove("Alle Jahre")
         jahre_liste.insert(0, "Alle Jahre")

    selected_index = window.einkauf_filter_jahr.get_selected()
    selected_value = None
    if selected_index > 0 and window.einkauf_filter_jahr.get_model():
        selected_value = window.einkauf_filter_jahr.get_model().get_string(selected_index)

    window.einkauf_filter_jahr.set_model(Gtk.StringList.new(jahre_liste))
    
    if selected_value and selected_value in jahre_liste:
        window.einkauf_filter_jahr.set_selected(jahre_liste.index(selected_value))
    else:
        window.einkauf_filter_jahr.set_selected(0)
        
def refresh_einkauf_list(window):
    if not hasattr(window, 'einkauf_list_box'): return

    while window.einkauf_list_box.get_row_at_index(0) is not None:
        window.einkauf_list_box.remove(window.einkauf_list_box.get_row_at_index(0))
    
    filter_text = window.einkauf_filter.get_text().lower()
    
    model_jahr = window.einkauf_filter_jahr.get_model()
    selected_jahr_index = window.einkauf_filter_jahr.get_selected()
    filter_jahr = model_jahr.get_string(selected_jahr_index) if model_jahr and selected_jahr_index != Gtk.INVALID_LIST_POSITION else "Alle Jahre"
    
    selected_monat_index = window.einkauf_filter_monat.get_selected()
    filter_monat = f"{selected_monat_index:02d}" if selected_monat_index > 0 else "Alle Monate"

    for einkauf in reversed(window.app.data['einkäufe']):
        if filter_jahr != "Alle Jahre" and not einkauf['datum'].startswith(filter_jahr):
            continue
        if filter_monat != "Alle Monate" and einkauf['datum'][5:7] != filter_monat:
            continue

        re_nr = einkauf.get('rechnungs_nr', '').lower()
        lieferant = einkauf.get('lieferant', '').lower()
        if filter_text and not (filter_text in einkauf['artikel'].lower() or filter_text in re_nr or filter_text in lieferant):
            continue

        row = Gtk.ListBoxRow()
        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5)
        box.set_margin_start(10); box.set_margin_end(10); box.set_margin_top(10); box.set_margin_bottom(10)
        header_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL)
        title = Gtk.Label(label=f"{einkauf['artikel']} - {einkauf['menge']} {einkauf['einheit']}"); title.add_css_class('heading')
        title.set_halign(Gtk.Align.START); title.set_hexpand(True); header_box.append(title)
        
        total_gross = einkauf.get('gesamt_brutto', einkauf.get('gesamt', 0.0))
        preis_label = Gtk.Label(label=f"{total_gross:.2f} €"); preis_label.add_css_class('title-3'); header_box.append(preis_label)
        box.append(header_box)
        
        re_nr_text = f"Re-Nr: {einkauf.get('rechnungs_nr', 'k.A.')}"

        unit_netto = einkauf.get('preis_netto', einkauf.get('preis', 0.0))
        unit_brutto = einkauf.get('preis_brutto', unit_netto * (1 + einkauf.get('steuer', 19)/100.0))
        price_details = f"{unit_netto:.4f}€ Netto / {unit_brutto:.4f}€ Brutto"

        details_text = f"{einkauf['datum']} | {re_nr_text} | {price_details} | {einkauf.get('lieferant', '')}"
        details = Gtk.Label(label=details_text)
        details.add_css_class('dim-label'); details.set_halign(Gtk.Align.START)
        box.append(details)

        # Buttons für Bearbeiten und Löschen
        button_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
        btn_edit = Gtk.Button(label="Bearbeiten")
        btn_edit.add_css_class('flat')
        btn_edit.connect('clicked', partial(open_einkauf_editor, window, einkauf))
        btn_del = Gtk.Button(label="Löschen")
        btn_del.add_css_class('destructive-action')
        btn_del.add_css_class('flat')
        btn_del.connect('clicked', partial(delete_einkauf, window, einkauf))
        button_box.append(btn_edit)
        button_box.append(btn_del)
        box.append(button_box)

        row.set_child(box)
        window.einkauf_list_box.append(row)

def open_einkauf_editor(window, einkauf, _btn=None):
    dlg = Adw.MessageDialog.new(window, "Einkauf bearbeiten", "Bearbeiten Sie den Einkauf.")
    dlg.add_response("cancel", "Abbrechen")
    dlg.add_response("save", "Speichern")
    dlg.set_response_appearance("save", Adw.ResponseAppearance.SUGGESTED)

    content = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=12)
    content.set_margin_start(16); content.set_margin_end(16); content.set_margin_top(16); content.set_margin_bottom(16)

    def create_dialog_row(label_text, widget):
        row_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
        label = Gtk.Label(label=label_text)
        label.set_size_request(150, -1); label.set_halign(Gtk.Align.START)
        row_box.append(label)
        if isinstance(widget, list):
            for w in widget:
                row_box.append(w)
        else:
            row_box.append(widget)
        content.append(row_box)

    # Lokale Variablen für Preis-Updates
    updating = False

    def on_netto_changed(netto_entry, steuer_dd, brutto_entry):
        nonlocal updating
        if updating: return
        updating = True
        try:
            netto = float(netto_entry.get_text().replace(',', '.'))
            steuer_idx = steuer_dd.get_selected()
            steuer_satz = 0.07 if steuer_idx == 0 else 0.19
            brutto = netto * (1 + steuer_satz)
            brutto_entry.set_text(f"{brutto:.4f}".replace('.', ','))
        except ValueError:
            brutto_entry.set_text("")
        updating = False

    def on_brutto_changed(brutto_entry, steuer_dd, netto_entry):
        nonlocal updating
        if updating: return
        updating = True
        try:
            brutto = float(brutto_entry.get_text().replace(',', '.'))
            steuer_idx = steuer_dd.get_selected()
            steuer_satz = 0.07 if steuer_idx == 0 else 0.19
            netto = brutto / (1 + steuer_satz)
            netto_entry.set_text(f"{netto:.4f}".replace('.', ','))
        except ValueError:
            netto_entry.set_text("")
        updating = False

    def on_steuer_changed(steuer_dd, netto_entry, brutto_entry):
        on_netto_changed(netto_entry, steuer_dd, brutto_entry)

    # Felder erstellen mit Defaults
    datum_entry = Gtk.Entry(); datum_entry.set_text(einkauf.get('datum', ''))
    create_dialog_row("Datum:", datum_entry)

    re_entry = Gtk.Entry(); re_entry.set_text(einkauf.get('rechnungs_nr', ''))
    create_dialog_row("Rechnungs-Nr.:", re_entry)

    artikel_namen = [a['name'] for a in window.app.data['stammdaten']['artikel']]
    dd_artikel_model = Gtk.StringList.new(artikel_namen)
    dd_artikel = Gtk.DropDown(); dd_artikel.set_model(dd_artikel_model); dd_artikel.set_enable_search(True)
    try:
        dd_artikel.set_selected(artikel_namen.index(einkauf['artikel']))
    except (ValueError, KeyError):
        pass
    create_dialog_row("Artikel:", dd_artikel)

    menge_spin = Gtk.SpinButton(); menge_spin.set_adjustment(Gtk.Adjustment(value=einkauf.get('menge', 1.0), lower=0.0001, upper=1000000, step_increment=1))
    einheit_dd = Gtk.DropDown.new_from_strings(EINHEITEN)
    try:
        einheit_dd.set_selected(EINHEITEN.index(einkauf.get('einheit', 'Stück')))
    except ValueError:
        einheit_dd.set_selected(0)
    create_dialog_row("Menge:", [menge_spin, einheit_dd])

    steuer_dd = Gtk.DropDown.new_from_strings(["7%", "19%"])
    steuer = einkauf.get('steuer', 19)
    steuer_dd.set_selected(0 if steuer == 7 else 1)
    create_dialog_row("Steuersatz:", steuer_dd)

    preis_box_widgets = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=5)
    unit_netto = einkauf.get('preis_netto', einkauf.get('preis', 0.0))
    netto_entry = Gtk.Entry(); netto_entry.set_text(f"{unit_netto:.4f}".replace('.', ','))
    netto_entry.set_hexpand(True)
    unit_brutto = einkauf.get('preis_brutto', unit_netto * (1 + steuer/100.0))
    brutto_entry = Gtk.Entry(); brutto_entry.set_text(f"{unit_brutto:.4f}".replace('.', ','))
    brutto_entry.set_hexpand(True)
    preis_box_widgets.append(netto_entry)
    preis_box_widgets.append(Gtk.Label(label="€ Netto /"))
    preis_box_widgets.append(brutto_entry)
    preis_box_widgets.append(Gtk.Label(label="€ Brutto"))
    netto_entry.connect('changed', lambda e: on_netto_changed(e, steuer_dd, brutto_entry))
    brutto_entry.connect('changed', lambda e: on_brutto_changed(e, steuer_dd, netto_entry))
    steuer_dd.connect('notify::selected', lambda d, p: on_steuer_changed(d, netto_entry, brutto_entry))
    create_dialog_row("Preis pro Einheit:", preis_box_widgets)

    lieferanten_namen = [l['name'] for l in window.app.data['stammdaten']['lieferanten']]
    dd_liefer_model = Gtk.StringList.new(lieferanten_namen)
    dd_liefer = Gtk.DropDown(); dd_liefer.set_model(dd_liefer_model); dd_liefer.set_enable_search(True)
    try:
        dd_liefer.set_selected(lieferanten_namen.index(einkauf.get('lieferant', '')))
    except ValueError:
        pass
    create_dialog_row("Lieferant:", dd_liefer)

    dlg.set_extra_child(content)

    def on_resp(d, resp):
        if resp == "save":
            try:
                new_datum = datum_entry.get_text()
                new_re_nr = re_entry.get_text().strip()
                new_art_idx = dd_artikel.get_selected()
                if new_art_idx == Gtk.INVALID_LIST_POSITION:
                    show_message(window, "Fehler", "Bitte einen Artikel auswählen.")
                    return
                new_artikel = dd_artikel.get_model().get_string(new_art_idx)
                new_menge = float(menge_spin.get_value())
                new_preis_netto = float(netto_entry.get_text().replace(',', '.'))
                new_preis_brutto = float(brutto_entry.get_text().replace(',', '.'))
                new_steuer = 7 if steuer_dd.get_selected() == 0 else 19
                new_liefer_idx = dd_liefer.get_selected()
                if new_liefer_idx == Gtk.INVALID_LIST_POSITION:
                    show_message(window, "Fehler", "Bitte einen Lieferanten auswählen.")
                    return
                new_liefer = dd_liefer.get_model().get_string(new_liefer_idx)
                new_einheit = EINHEITEN[einheit_dd.get_selected()]

                # Alte Werte speichern
                old_artikel = einkauf['artikel']
                old_menge = einkauf.get('menge', 0.0)

                # Update Einkauf
                einkauf['datum'] = new_datum
                einkauf['rechnungs_nr'] = new_re_nr
                einkauf['artikel'] = new_artikel
                einkauf['menge'] = new_menge
                einkauf['einheit'] = new_einheit
                einkauf['preis_netto'] = new_preis_netto
                einkauf['preis_brutto'] = new_preis_brutto
                einkauf['steuer'] = new_steuer
                einkauf['lieferant'] = new_liefer
                einkauf['gesamt_brutto'] = round(new_menge * new_preis_brutto, 2)

                # Bestand anpassen
                if old_artikel != new_artikel:
                    # Alten Artikel entfernen
                    old_bestand = next((b for b in window.app.data['bestand'] if b['artikel'] == old_artikel), None)
                    if old_bestand:
                        old_bestand['menge'] -= old_menge
                        if old_bestand['menge'] <= 0:
                            window.app.data['bestand'].remove(old_bestand)
                    # Neuen Artikel hinzufügen/aktualisieren
                    new_bestand = next((b for b in window.app.data['bestand'] if b['artikel'] == new_artikel), None)
                    if not new_bestand:
                        new_bestand = {'artikel': new_artikel, 'menge': 0, 'preis': new_preis_netto, 'einheit': new_einheit}
                        window.app.data['bestand'].append(new_bestand)
                    new_bestand['menge'] += new_menge
                    new_bestand['preis'] = new_preis_netto
                else:
                    bestand_item = next((b for b in window.app.data['bestand'] if b['artikel'] == new_artikel), None)
                    if bestand_item:
                        bestand_item['menge'] += (new_menge - old_menge)
                        if new_menge > 0:
                            bestand_item['preis'] = new_preis_netto
                        if bestand_item['menge'] <= 0:
                            window.app.data['bestand'].remove(bestand_item)

                window.app.save_data()
                refresh_einkauf_list(window)
                from bestand import refresh_bestand_list
                refresh_bestand_list(window)
                from kassenbuch import refresh_kassenbuch
                refresh_kassenbuch(window)
                show_message(window, "Erfolg", "Einkauf aktualisiert")
            except (ValueError, TypeError):
                show_message(window, "Fehler", "Bitte gültige Werte eingeben.")
        d.close()

    dlg.connect("response", on_resp)
    dlg.present()

def delete_einkauf(window, einkauf, _btn=None):
    if confirm(window, "Löschen bestätigen", f"Einkauf vom {einkauf.get('datum', '')} wirklich löschen?"):
        index = window.app.data['einkäufe'].index(einkauf)
        del window.app.data['einkäufe'][index]
        # Bestand anpassen
        bestand_item = next((b for b in window.app.data['bestand'] if b['artikel'] == einkauf['artikel']), None)
        if bestand_item:
            bestand_item['menge'] -= einkauf.get('menge', 0.0)
            if bestand_item['menge'] <= 0:
                window.app.data['bestand'].remove(bestand_item)
        window.app.save_data()
        update_einkauf_jahr_filter(window)
        refresh_einkauf_list(window)
        from bestand import refresh_bestand_list
        refresh_bestand_list(window)
        from kassenbuch import refresh_kassenbuch
        refresh_kassenbuch(window)
        show_message(window, "Erfolg", "Einkauf gelöscht")