import datetime
from pathlib import Path
import facturx
from utils import show_message
from io import BytesIO
from reportlab.pdfgen import canvas
from reportlab.lib.pagesizes import A4
from reportlab.lib.units import mm
from reportlab.pdfbase.pdfmetrics import stringWidth
import xml.etree.ElementTree as ET
import re

# ============================================
#  EINHEITEN-MAPPING (UN/CEFACT Codes)
# ============================================
EINHEITEN_MAP = {
    "Stück": "C62",  # "one" / "unit"
    "kg": "KGM",
    "g": "GRM",
    "L": "LTR",
}

# ============================================
#  Hilfsfunktionen (PDF)
# ============================================
def _right(c: canvas.Canvas, text: str, x_right: float, y: float, font="Helvetica", size=10):
    c.setFont(font, size)
    w = stringWidth(text, font, size)
    c.drawString(x_right - w, y, text)

def _format_eur(val: float) -> str:
    return f"{val:,.2f} €".replace(",", "X").replace(".", ",").replace("X", ".")

def _parse_zahlungsziel(zahlungsziel: str, rechnungsdatum: datetime.date):
    """
    Versucht, ein Fälligkeitsdatum + Label aus dem Zahlungsziel herzuleiten.
    - '14 Tage netto' -> rechnungsdatum + 14 Tage
    - 'YYYY-MM-DD'    -> dieses Datum
    - sonst -> None (wir zeigen nur den Text)
    """
    if not zahlungsziel:
        return None, None
    # yyyy-mm-dd
    m = re.match(r"^\s*(\d{4})-(\d{2})-(\d{2})\s*$", zahlungsziel)
    if m:
        try:
            due = datetime.date(int(m.group(1)), int(m.group(2)), int(m.group(3)))
            return due, f"fällig bis: {due.isoformat()}"
        except Exception:
            pass
    # '14 Tage', '30 Tage netto', etc.
    m = re.search(r"(\d+)\s*Tage", zahlungsziel, re.IGNORECASE)
    if m:
        try:
            days = int(m.group(1))
            due = rechnungsdatum + datetime.timedelta(days=days)
            return due, f"fällig bis: {due.isoformat()} (in {days} Tagen)"
        except Exception:
            pass
    return None, None

# ============================================
#  Factur-X XML (BASIC) minimal erzeugen – inkl. Note (Zahlungsziel)
# ============================================
def build_facturx_basic_xml(invoice_dict, note_text: str | None = None) -> bytes:
    NS = {
        "rsm": "urn:un:unece:uncefact:data:standard:CrossIndustryInvoice:100",
        "ram": "urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100",
        "qdt": "urn:un:unece:uncefact:data:standard:QualifiedDataType:100",
        "udt": "urn:un:unece:uncefact:data:standard:UnqualifiedDataType:100",
    }
    for p, uri in NS.items():
        ET.register_namespace(p, uri)

    rsm = ET.Element(ET.QName(NS["rsm"], "CrossIndustryInvoice"))

    # Kopf
    exh = ET.SubElement(rsm, ET.QName(NS["rsm"], "ExchangedDocument"))
    ET.SubElement(exh, ET.QName(NS["ram"], "ID")).text = invoice_dict["invoice_id"]
    ET.SubElement(exh, ET.QName(NS["ram"], "TypeCode")).text = invoice_dict.get("doc_type_code", "380")
    issue_dt = ET.SubElement(exh, ET.QName(NS["ram"], "IssueDateTime"))
    dt_str = ET.SubElement(issue_dt, ET.QName(NS["udt"], "DateTimeString"), attrib={"format": "102"})
    inv_date = invoice_dict["invoice_date"]
    dt_str.text = inv_date.strftime("%Y%m%d") if isinstance(inv_date, (datetime.date, datetime.datetime)) else str(inv_date)

    # optionale Note (Zahlungsziel)
    if note_text:
        note = ET.SubElement(exh, ET.QName(NS["ram"], "IncludedNote"))
        ET.SubElement(note, ET.QName(NS["ram"], "Content")).text = note_text

    # Transaktion
    trad_tx = ET.SubElement(rsm, ET.QName(NS["rsm"], "SupplyChainTradeTransaction"))
    trd_agr = ET.SubElement(trad_tx, ET.QName(NS["ram"], "ApplicableHeaderTradeAgreement"))

    # Verkäufer
    seller = invoice_dict["seller_dict"]
    seller_node = ET.SubElement(trd_agr, ET.QName(NS["ram"], "SellerTradeParty"))
    ET.SubElement(seller_node, ET.QName(NS["ram"], "Name")).text = seller.get("name", "")
    s_addr = ET.SubElement(seller_node, ET.QName(NS["ram"], "PostalTradeAddress"))
    for tag, key in [("PostcodeCode", "postcode"), ("LineOne", "street"), ("CityName", "city"), ("CountryID", "country_id")]:
        ET.SubElement(s_addr, ET.QName(NS["ram"], tag)).text = seller.get(key, "")

    # Käufer
    buyer = invoice_dict["buyer_dict"]
    buyer_node = ET.SubElement(trd_agr, ET.QName(NS["ram"], "BuyerTradeParty"))
    ET.SubElement(buyer_node, ET.QName(NS["ram"], "Name")).text = buyer.get("name", "")
    b_addr = ET.SubElement(buyer_node, ET.QName(NS["ram"], "PostalTradeAddress"))
    for tag, key in [("PostcodeCode", "postcode"), ("LineOne", "street"), ("CityName", "city"), ("CountryID", "country_id")]:
        ET.SubElement(b_addr, ET.QName(NS["ram"], tag)).text = buyer.get(key, "")

    # Währung
    ET.SubElement(trad_tx, ET.QName(NS["ram"], "ApplicableHeaderTradeDelivery"))
    trd_set = ET.SubElement(trad_tx, ET.QName(NS["ram"], "ApplicableHeaderTradeSettlement"))
    ET.SubElement(trd_set, ET.QName(NS["ram"], "InvoiceCurrencyCode")).text = invoice_dict.get("currency", "EUR")

    # Positionen
    for i, line in enumerate(invoice_dict["lines"], start=1):
        line_node = ET.SubElement(trad_tx, ET.QName(NS["ram"], "IncludedSupplyChainTradeLineItem"))
        trad_prod = ET.SubElement(line_node, ET.QName(NS["ram"], "SpecifiedTradeProduct"))
        ET.SubElement(trad_prod, ET.QName(NS["ram"], "Name")).text = str(line.get("item_name", f"Position {i}"))

        trade_agreement = ET.SubElement(line_node, ET.QName(NS["ram"], "SpecifiedLineTradeAgreement"))
        gross_price = ET.SubElement(trade_agreement, ET.QName(NS["ram"], "GrossPriceProductTradePrice"))
        ET.SubElement(gross_price, ET.QName(NS["ram"], "ChargeAmount")).text = f"{float(line.get('net_price', 0.0)):.2f}"

        trade_delivery = ET.SubElement(line_node, ET.QName(NS["ram"], "SpecifiedLineTradeDelivery"))
        qty = ET.SubElement(trade_delivery, ET.QName(NS["ram"], "BilledQuantity"), attrib={"unitCode": line.get("unit_code", "C62")})
        qty.text = f"{float(line.get('quantity', 0.0)):.4f}"

        trade_settlement = ET.SubElement(line_node, ET.QName(NS["ram"], "SpecifiedLineTradeSettlement"))
        tax = ET.SubElement(trade_settlement, ET.QName(NS["ram"], "ApplicableTradeTax"))
        ET.SubElement(tax, ET.QName(NS["ram"], "CategoryCode")).text = line.get("tax_category_code", "S")
        ET.SubElement(tax, ET.QName(NS["ram"], "TypeCode")).text = line.get("tax_type_code", "VAT")
        ET.SubElement(tax, ET.QName(NS["ram"], "RateApplicablePercent")).text = str(line.get("tax_rate", 19))

        net_amount = float(line.get("net_price", 0.0)) * float(line.get("quantity", 0.0))
        line_monetary = ET.SubElement(trade_settlement, ET.QName(NS["ram"], "SpecifiedTradeSettlementLineMonetarySummation"))
        ET.SubElement(line_monetary, ET.QName(NS["ram"], "LineTotalAmount")).text = f"{net_amount:.2f}"

    # Summen (minimal)
    total_net = sum(float(l.get("net_price", 0.0)) * float(l.get("quantity", 0.0)) for l in invoice_dict["lines"])
    mon_sum = ET.SubElement(trd_set, ET.QName(NS["ram"], "SpecifiedTradeSettlementHeaderMonetarySummation"))
    ET.SubElement(mon_sum, ET.QName(NS["ram"], "LineTotalAmount")).text = f"{total_net:.2f}"
    ET.SubElement(mon_sum, ET.QName(NS["ram"], "TaxBasisTotalAmount")).text = f"{total_net:.2f}"

    return ET.tostring(rsm, encoding="utf-8", xml_declaration=True)

# ============================================
#  Tabellen-Rendering
# ============================================
def _draw_items_table(c: canvas.Canvas, x: float, y: float, width: float, lines: list, font="Helvetica", size=10):
    col_w = [8*mm, 86*mm, 15*mm, 12*mm, 18*mm, 15*mm, 16*mm]
    headers = ["Pos", "Artikel", "Menge", "Einh.", "EP netto", "Netto", "Brutto"]
    row_h = 8*mm

    c.setFont("Helvetica-Bold", 10)
    cx = x
    for i, h in enumerate(headers):
        if i in (4, 5, 6):
            _right(c, h, cx + col_w[i], y, "Helvetica-Bold", 10)
        else:
            c.drawString(cx, y, h)
        cx += col_w[i]
    y -= row_h
    c.setFont(font, size)

    page_bottom = 20*mm
    for idx, line in enumerate(lines, start=1):
        if y < page_bottom:
            c.showPage()
            width_page, height_page = A4
            y = height_page - 30*mm
            c.setFont("Helvetica-Bold", 12)
            c.drawString(20*mm, y, "Fortsetzung Positionen")
            y -= 10*mm
            c.setFont("Helvetica-Bold", 10)
            cx = x
            for i, h in enumerate(headers):
                if i in (4, 5, 6):
                    _right(c, h, cx + col_w[i], y, "Helvetica-Bold", 10)
                else:
                    c.drawString(cx, y, h)
                cx += col_w[i]
            y -= row_h
            c.setFont(font, size)

        menge = float(line.get("quantity", 0.0))
        unit_code = line.get("unit_code", "C62")
        display_unit = next((k for k, v in EINHEITEN_MAP.items() if v == unit_code), unit_code)
        ep_netto = float(line.get("net_price", 0.0))
        steuer = float(line.get("tax_rate", 19.0))
        zeilen_netto = menge * ep_netto
        zeilen_brutto = zeilen_netto * (1.0 + steuer/100.0)

        cx = x
        c.drawString(cx, y, str(idx)); cx += col_w[0]
        artikel = str(line.get("item_name", ""))
        max_w = col_w[1] - 1*mm
        while stringWidth(artikel, font, size) > max_w and len(artikel) > 3:
            artikel = artikel[:-4] + "…"
        c.drawString(cx, y, artikel); cx += col_w[1]
        _right(c, f"{menge:g}", cx + col_w[2], y); cx += col_w[2]
        c.drawString(cx, y, display_unit); cx += col_w[3]
        _right(c, _format_eur(ep_netto), cx + col_w[4], y); cx += col_w[4]
        _right(c, _format_eur(zeilen_netto), cx + col_w[5], y); cx += col_w[5]
        _right(c, _format_eur(zeilen_brutto), cx + col_w[6], y); cx += col_w[6]
        y -= row_h

    c.line(x, y + row_h*0.3, x + sum(col_w), y + row_h*0.3)
    return y

def _draw_totals(c: canvas.Canvas, x: float, y: float, width: float, lines: list, font="Helvetica", size=11):
    subtotal_net = 0.0
    taxes_by_rate = {}
    for line in lines:
        qty = float(line.get("quantity", 0.0))
        unit_price = float(line.get("net_price", 0.0))
        rate = float(line.get("tax_rate", 19.0))
        net = qty * unit_price
        tax = net * rate / 100.0
        subtotal_net += net
        taxes_by_rate[rate] = taxes_by_rate.get(rate, 0.0) + tax

    total_tax = sum(taxes_by_rate.values())
    total_gross = subtotal_net + total_tax

    right = x + width
    line_h = 7*mm
    c.setFont("Helvetica-Bold", size)
    _right(c, "Zwischensumme (netto):", right - 60*mm, y)
    _right(c, _format_eur(subtotal_net), right, y)
    y -= line_h

    c.setFont("Helvetica", size)
    for rate in sorted(taxes_by_rate.keys()):
        _right(c, f"USt {int(rate)}%:", right - 60*mm, y)
        _right(c, _format_eur(taxes_by_rate[rate]), right, y)
        y -= line_h

    c.setFont("Helvetica-Bold", size+1)
    _right(c, "Gesamtsumme (brutto):", right - 60*mm, y)
    _right(c, _format_eur(total_gross), right, y)
    y -= line_h

    return y

# ============================================
#  Kompat-Wrapper (schreibt notfalls selbst)
# ============================================
def _embed_facturx_xml_into_pdf(pdf_bytes: bytes, xml_bytes: bytes, out_path: Path) -> None:
    result = None
    try:
        result = facturx.generate_from_binary(
            pdf_bytes, xml_bytes, str(out_path),
            level="BASIC", check_xsd=False, pdfa=True,
        )
    except TypeError:
        try:
            result = facturx.generate_from_binary(
                pdf_bytes, xml_bytes, str(out_path),
                level="BASIC", check_xsd=False,
            )
        except TypeError:
            try:
                result = facturx.generate_from_binary(
                    pdf_bytes, xml_bytes, str(out_path),
                    facturx_level="BASIC", check_xsd=False,
                )
            except (TypeError, AttributeError):
                result = None
    except AttributeError:
        result = None

    if result is None:
        try:
            result = facturx.generate_facturx_from_binary(
                pdf_bytes, xml_bytes, str(out_path),
                level="BASIC", check_xsd=False,
            )
        except TypeError:
            result = facturx.generate_facturx_from_binary(
                pdf_bytes, xml_bytes, str(out_path),
                level="BASIC",
            )

    if isinstance(result, (bytes, bytearray)) and not out_path.exists():
        out_path.write_bytes(result)

# ============================================
#  Hauptfunktion
# ============================================
def export_rechnung_zugferd(window, rechnung_nr, _btn=None):
    """Erstellt eine ZUGFeRD/Factur-X BASIC-Rechnung, zeichnet Positionen & Kopf und speichert die Datei."""
    try:
        # 1) Daten sammeln
        items = [v for v in window.app.data["verkäufe"] if v.get("rechnung_nr") == rechnung_nr]
        if not items:
            show_message(window, "Fehler", "Keine Positionen für diese Rechnung gefunden.")
            return

        kunde = next((k for k in window.app.data["stammdaten"]["kunden"] if k["name"] == items[0]["kunde"]), {})
        firma = window.app.data["stammdaten"]["firma"]
        base = Path(__file__).resolve().parent
        logo_path = base / "icons" / "logo.png"

        rechnungs_datum_str = sorted([i["datum"] for i in items])[0]
        rechnungs_datum = datetime.datetime.strptime(rechnungs_datum_str, "%Y-%m-%d").date()
        zahlungsziel = next((i.get("zahlungsziel") for i in items if i.get("zahlungsziel")), "") or ""

        # Fälligkeitsdatum ableiten (wenn möglich)
        due_date, due_label = _parse_zahlungsziel(zahlungsziel, rechnungs_datum)

        # 2) Lines aufbereiten
        lines = []
        for v in items:
            lines.append({
                "item_name": v.get("artikel", "Unbekannter Artikel"),
                "quantity": float(v.get("menge", 0)),
                "unit_code": EINHEITEN_MAP.get(v.get("einheit", "Stück"), "C62"),
                "net_price": float(v.get("preis", 0)),
                "tax_type_code": "VAT",
                "tax_category_code": "S",
                "tax_rate": float(v.get("steuer", 19)),
            })

        invoice_dict = {
            "doc_type_code": "380",
            "invoice_id": str(rechnung_nr),
            "invoice_date": rechnungs_datum,
            "currency": "EUR",
            "seller_dict": {
                "name": firma.get("name"),
                "street": firma.get("strasse"),
                "city": firma.get("ort"),
                "postcode": firma.get("plz"),
                "country_id": "DE",
            },
            "buyer_dict": {
                "name": kunde.get("name"),
                "street": kunde.get("strasse"),
                "city": kunde.get("ort"),
                "postcode": kunde.get("plz"),
                "country_id": "DE",
            },
            "lines": lines,
        }

        # 3) Factur-X XML (Zahlungsziel als Note)
        note_parts = []
        if zahlungsziel:
            note_parts.append(f"Zahlungsziel: {zahlungsziel}")
        if due_label:
            note_parts.append(due_label)
        note = " | ".join(note_parts) if note_parts else None
        facturx_xml_content = build_facturx_basic_xml(invoice_dict, note_text=note)

        # 4) Visuelles PDF (mit Firmenadresse rechts, Logo, Zahlungsziel, Bank) erzeugen
        pdf_buffer = BytesIO()
        c = canvas.Canvas(pdf_buffer, pagesize=A4)
        page_w, page_h = A4
        margin = 20*mm
        width = page_w - 2*margin
        y = page_h - margin

        # Logo links oben
        if logo_path.exists():
            c.drawImage(
                str(logo_path),
                margin, page_h - margin - 25*mm,
                width=35*mm, height=25*mm,
                preserveAspectRatio=True, mask="auto"
            )

        # Firmenadresse OBEN RECHTS
        right = page_w - margin
        c.setFont("Helvetica-Bold", 12)
        _right(c, firma.get("name", ""), right, page_h - margin - 5*mm)
        c.setFont("Helvetica", 10)
        _right(c, f"{firma.get('strasse','')}", right, page_h - margin - 11*mm)
        _right(c, f"{firma.get('plz','')} {firma.get('ort','')}", right, page_h - margin - 16*mm)
        if firma.get("email"):
            _right(c, f"Email: {firma.get('email','')}", right, page_h - margin - 22*mm)
        if firma.get("telefon"):
            _right(c, f"Tel: {firma.get('telefon','')}", right, page_h - margin - 27*mm)

        # Titel und Meta links
        y = page_h - margin - 35*mm
        c.setFont("Helvetica-Bold", 16)
        c.drawString(margin, y, f"Rechnung {rechnung_nr}")
        y -= 7*mm
        c.setFont("Helvetica", 10)
        c.drawString(margin, y, f"Datum: {rechnungs_datum_str}")
        if zahlungsziel:
            y -= 5*mm
            # Zeige Zahlungsziel-Zeile (inkl. evtl. fällig-bis)
            zz_line = f"Zahlungsziel: {zahlungsziel}"
            if due_label:
                zz_line += f" ({due_label})"
            c.drawString(margin, y, zz_line)

        # Kunde
        y -= 12*mm
        c.setFont("Helvetica-Bold", 11)
        c.drawString(margin, y, "Rechnung an:")
        y -= 5*mm
        c.setFont("Helvetica", 10)
        c.drawString(margin, y, kunde.get("name", ""))
        y -= 5*mm
        c.drawString(margin, y, kunde.get("strasse", ""))
        y -= 5*mm
        c.drawString(margin, y, f"{kunde.get('plz','')} {kunde.get('ort','')}")

        # Tabelle
        y -= 12*mm
        y = _draw_items_table(c, margin, y, width, lines)

        # Summen
        y -= 4*mm
        y = _draw_totals(c, margin, y, width, lines)

        # Zahlungsziel & Bankverbindung unten gut sichtbar wiederholen
        y -= 10*mm
        if y < 30*mm:
            c.showPage()
            page_w, page_h = A4
            margin = 20*mm
            width = page_w - 2*margin
            y = page_h - margin

        c.setFont("Helvetica-Bold", 10)
        if zahlungsziel:
            zz_line2 = f"Zahlungsziel: {zahlungsziel}"
            if due_label:
                zz_line2 += f" ({due_label})"
            c.drawString(margin, y, zz_line2)
            y -= 6*mm

        c.setFont("Helvetica", 9)
        bank_parts = []
        if firma.get("bank"): bank_parts.append(f"Bank: {firma.get('bank')}")
        if firma.get("iban"): bank_parts.append(f"IBAN: {firma.get('iban')}")
        if firma.get("bic"):  bank_parts.append(f"BIC: {firma.get('bic')}")
        bank_str = " • ".join(bank_parts) if bank_parts else ""
        if bank_str:
            c.drawString(margin, y, bank_str)

        c.save()
        pdf_buffer.seek(0)
        original_pdf_bytes = pdf_buffer.read()

        # 5) XML + PDF zusammenführen
        out_dir = (Path(__file__).resolve().parent / "rechnungen").absolute()
        out_dir.mkdir(parents=True, exist_ok=True)
        out_file = (out_dir / f"ZUGFeRD_Rechnung_{rechnung_nr}.pdf").absolute()

        _embed_facturx_xml_into_pdf(original_pdf_bytes, facturx_xml_content, out_file)

        if out_file.exists() and out_file.stat().st_size > 0:
            show_message(window, "ZUGFeRD-PDF erstellt", f"Gespeichert unter:\n{out_file}")
        else:
            show_message(window, "Hinweis", f"Die Bibliothek hat keine Datei geschrieben.\nErwarteter Speicherort:\n{out_file}")

    except Exception as e:
        import traceback
        traceback.print_exc()
        show_message(window, "Fehler beim ZUGFeRD-Export", f"Ein Fehler ist aufgetreten:\n{e}")
