from pathlib import Path
from utils import Gtk, Adw, EINHEITEN, show_message, confirm
from gi.repository import Gio, GLib
from einkauf import create_einkauf_page, create_einkauf_list_page, refresh_einkauf_list, update_einkauf_jahr_filter
from verkauf import create_verkauf_page, create_verkauf_list_page, refresh_verkauf_list, update_verkauf_jahr_filter
from bestand import create_bestand_page, refresh_bestand_list
from kassenbuch import create_kassenbuch_page, refresh_kassenbuch
from stammdaten import create_stammdaten_page, refresh_stammdaten_lists, update_master_data_dropdowns
from rechnung import (
    create_rechnung_page,
    create_rechnung_list_page,
    refresh_rechnung_list,
    update_rechnung_jahr_filter,
)
from sperrliste import create_sperrliste_page, refresh_sperrliste_list


class KassenbuchWindow(Adw.ApplicationWindow):
    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        # --- App-Icon setzen ---
        self.set_icon_name("app_icon")

        self.set_default_size(1200, 860)
        self.set_resizable(True)
        self.app = self.get_application()
        self._is_updating_prices = False

        # Toolbar + HeaderBar
        self.toolbar_view = Adw.ToolbarView()
        header = Adw.HeaderBar()
        header.set_hexpand(True)
        header.set_show_start_title_buttons(False)
        header.set_show_end_title_buttons(True)

        # Icon links in der Titelleiste
        euro_icon = Gtk.Image.new_from_icon_name("app_icon")
        euro_icon.set_pixel_size(24)
        euro_icon.set_valign(Gtk.Align.CENTER)
        euro_icon.set_margin_start(6)
        header.pack_start(euro_icon)

        title = Adw.WindowTitle(title="Kassenbuch", subtitle="")
        header.set_title_widget(title)

        # Menü (Über Kassenbuch & Backup/Restore)
        menu_button = Gtk.MenuButton()
        menu_button.set_icon_name("open-menu-symbolic")
        menu_model = Gio.Menu()
        menu_model.append("Über Kassenbuch", "app.about")
        menu_model.append("Backup erstellen", "app.backup")
        menu_model.append("Backup wiederherstellen", "app.restore")
        menu_button.set_menu_model(menu_model)
        header.pack_end(menu_button)

        # Aktion für "Über"
        about_action = Gio.SimpleAction.new("about", None)
        about_action.connect("activate", self.on_about_clicked)
        self.get_application().add_action(about_action)
        
        # Aktion für "Backup"
        backup_action = Gio.SimpleAction.new("backup", None)
        backup_action.connect("activate", self.on_backup_clicked)
        self.get_application().add_action(backup_action)
        
        # Aktion für "Restore"
        restore_action = Gio.SimpleAction.new("restore", None)
        restore_action.connect("activate", self.on_restore_clicked)
        self.get_application().add_action(restore_action)

        self.toolbar_view.add_top_bar(header)

        # Hauptlayout
        self.main_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL)
        self.toolbar_view.set_content(self.main_box)
        self.set_content(self.toolbar_view)

        # Sidebar + Inhaltsbereich
        self.create_sidebar()

        self.content_stack = Gtk.Stack()
        self.content_stack.set_transition_type(Gtk.StackTransitionType.SLIDE_LEFT_RIGHT)
        self.main_box.append(self.content_stack)

        # Seiten erstellen
        create_einkauf_page(self)
        create_einkauf_list_page(self)
        create_bestand_page(self)
        create_verkauf_page(self)
        create_verkauf_list_page(self)
        create_kassenbuch_page(self)
        create_stammdaten_page(self)
        create_rechnung_page(self)
        create_rechnung_list_page(self)
        create_sperrliste_page(self)

        # Dropdowns + Listen initialisieren
        update_master_data_dropdowns(self)
        update_einkauf_jahr_filter(self)
        update_verkauf_jahr_filter(self) # NEU
        refresh_einkauf_list(self)
        refresh_bestand_list(self)
        refresh_verkauf_list(self)
        refresh_kassenbuch(self)
        refresh_stammdaten_lists(self)
        refresh_sperrliste_list(self)
        update_rechnung_jahr_filter(self)
        refresh_rechnung_list(self)

        # Standardansicht
        self.content_stack.set_visible_child_name("kassenbuch")

    def create_sidebar(self):
        sidebar = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=0)
        sidebar.set_size_request(260, -1)
        sidebar.add_css_class("sidebar")

        header = Gtk.Label(label="Kassenbuch")
        header.add_css_class("title-1")
        header.set_margin_top(20)
        header.set_margin_bottom(20)
        sidebar.append(header)

        list_box = Gtk.ListBox()
        list_box.set_selection_mode(Gtk.SelectionMode.SINGLE)
        list_box.add_css_class("navigation-sidebar")

        pages = [
            ("Einkauf erfassen", "einkauf"),
            ("Einkäufe anzeigen", "einkauf_list"),
            ("Bestandsliste", "bestand"),
            ("Verkauf erfassen", "verkauf"),
            ("Verkäufe anzeigen", "verkauf_list"),
            ("Kassenbuch", "kassenbuch"),
            ("Sperrliste", "sperrliste"),
            ("Rechnung erstellen", "rechnung"),
            ("Rechnungen", "rechnung_list"),
            ("Stammdaten", "stammdaten"),
        ]

        for label, page_name in pages:
            row = Gtk.ListBoxRow()
            box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
            box.set_margin_start(10)
            box.set_margin_end(10)
            box.set_margin_top(8)
            box.set_margin_bottom(8)
            label_widget = Gtk.Label(label=label)
            label_widget.set_halign(Gtk.Align.START)
            box.append(label_widget)
            row.set_child(box)
            row.page_name = page_name
            list_box.append(row)

        list_box.connect("row-activated", self.on_sidebar_row_activated)
        sidebar.append(list_box)
        self.main_box.append(sidebar)

        # Standard: Kassenbuch vorausgewählt
        list_box.select_row(list_box.get_row_at_index(5))

    def on_sidebar_row_activated(self, list_box, row):
        self.content_stack.set_visible_child_name(row.page_name)

    def confirm(self, title, text):
        return confirm(self, title, text)

    def show_message(self, title, message):
        show_message(self, title, message)

    def on_about_clicked(self, action, param):
        """Zeigt das Info-Fenster an"""
        dialog = Adw.AboutWindow(transient_for=self)
        dialog.set_application_name("Kassenbuch")
        dialog.set_application_icon("app_icon")
        dialog.set_version("1.1")
        dialog.set_developer_name("designed by armin@pinguin-tv")
        dialog.set_website("https://www.pinguin-tv.de")
        dialog.add_link("Spenden via PayPal", "https://www.paypal.com/paypalme/pinguintv")
        dialog.set_license_type(Gtk.License.GPL_3_0_ONLY)
        dialog.set_copyright("© 2025 Armin")
        dialog.present()

    def on_backup_clicked(self, action, param):
        """Zeigt den Dialog zum Speichern eines Backups an."""
        from datetime import datetime
        
        dialog = Gtk.FileDialog.new()
        dialog.set_title("Backup speichern unter…")
        
        default_name = f"kassenbuch_backup_{datetime.now().strftime('%Y-%m-%d')}.db"
        dialog.set_initial_name(default_name)
        
        dialog.save(self, None, self.on_backup_save_response)

    def on_backup_save_response(self, dialog, result):
        """Wird aufgerufen, nachdem der Benutzer einen Speicherort ausgewählt hat."""
        import shutil

        try:
            dest_file = dialog.save_finish(result)
            dest_path = dest_file.get_path()
            source_path = self.app.db_path
            
            shutil.copy2(source_path, dest_path)
            
            self.show_message("Erfolg", f"Backup erfolgreich gespeichert unter:\n{dest_path}")
        
        except GLib.Error as e:
            if not e.matches(Gio.io_error_quark(), Gio.IOErrorEnum.CANCELLED):
                self.show_message("Fehler", f"Backup konnte nicht erstellt werden:\n{e}")
        except Exception as e:
            self.show_message("Fehler", f"Ein unerwarteter Fehler ist aufgetreten:\n{e}")

    def on_restore_clicked(self, action, param):
        """Zeigt den Dialog zum Auswählen einer Backup-Datei an."""
        dialog = Gtk.FileDialog.new()
        dialog.set_title("Backup-Datei zum Wiederherstellen auswählen")

        db_filter = Gtk.FileFilter.new()
        db_filter.add_pattern("*.db")
        filters = Gio.ListStore.new(Gtk.FileFilter)
        filters.append(db_filter)
        dialog.set_filters(filters)

        dialog.open(self, None, self.on_restore_open_response)

    def on_restore_open_response(self, dialog, result):
        """Wird aufgerufen, nachdem eine Datei zum Wiederherstellen ausgewählt wurde."""
        import shutil
        
        try:
            source_file = dialog.open_finish(result)
            source_path = source_file.get_path()
            dest_path = self.app.db_path

            warning_text = (
                f"Möchten Sie wirklich das Backup von\n\n<b>{source_file.get_basename()}</b>\n\nwiederherstellen?\n\n"
                "<b>ACHTUNG:</b> Alle aktuellen Daten werden unwiderruflich überschrieben!"
            )
            if not self.confirm("Wiederherstellung bestätigen", warning_text):
                return

            shutil.copy2(source_path, dest_path)
            
            self.show_message("Erfolg", "Das Backup wurde erfolgreich wiederhergestellt.\n\nDas Programm wird jetzt geschlossen. Bitte starten Sie es neu, um die Daten zu laden.")
            self.close()

        except GLib.Error as e:
            if not e.matches(Gio.io_error_quark(), Gio.IOErrorEnum.CANCELLED):
                self.show_message("Fehler", f"Backup konnte nicht wiederhergestellt werden:\n{e}")
        except Exception as e:
            self.show_message("Fehler", f"Ein unerwarteter Fehler ist aufgetreten:\n{e}")