from utils import Gtk, Adw, EINHEITEN, partial, show_message, confirm
from datetime import datetime
from db import next_rechnung_nr
from pathlib import Path
from zugferd_export import export_rechnung_zugferd

# PDF mit ReportLab
try:
    from reportlab.lib.pagesizes import A4
    from reportlab.lib.units import mm
    from reportlab.pdfgen import canvas
    REPORTLAB_OK = True
except Exception:
    REPORTLAB_OK = False


# =====================================================
# Seite: RECHNUNG ERSTELLEN (Einzel + mehrere Verkäufe)
# =====================================================

def create_rechnung_page(window):
    page = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=20)
    page.set_margin_start(40)
    page.set_margin_end(40)
    page.set_margin_top(40)
    page.set_margin_bottom(40)

    header = Gtk.Label(label="Rechnung erstellen")
    header.add_css_class("title-2")
    header.set_halign(Gtk.Align.START)
    page.append(header)

    # ---------- Einzel-Position ----------
    frame_single = Adw.PreferencesGroup(title="Einzelne Position")
    form = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=12)

    def row(label_text, widget):
        box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
        lab = Gtk.Label(label=label_text)
        lab.set_size_request(150, -1)
        lab.set_halign(Gtk.Align.START)
        box.append(lab)
        if isinstance(widget, list):
            for w in widget:
                box.append(w)
        else:
            box.append(widget)
        form.append(box)

    window.rech_datum = Gtk.Entry()
    window.rech_datum.set_text(datetime.now().strftime("%Y-%m-%d"))
    row("Datum:", window.rech_datum)

    kunden_namen = [k["name"] for k in window.app.data["stammdaten"]["kunden"]]
    window.rech_kunde_dd = Gtk.DropDown()
    window.rech_kunde_dd.set_model(Gtk.StringList.new(kunden_namen))
    window.rech_kunde_dd.set_enable_search(True)
    row("Kunde:", window.rech_kunde_dd)

    artikel_namen = [a["name"] for a in window.app.data["stammdaten"]["artikel"]]
    window.rech_artikel_dd = Gtk.DropDown()
    window.rech_artikel_dd.set_model(Gtk.StringList.new(artikel_namen))
    window.rech_artikel_dd.set_enable_search(True)
    row("Artikel:", window.rech_artikel_dd)

    window.rech_menge = Gtk.SpinButton()
    window.rech_menge.set_adjustment(Gtk.Adjustment(value=1, lower=0.0001, upper=1000000, step_increment=1))
    window.rech_einheit = Gtk.DropDown.new_from_strings(EINHEITEN)
    window.rech_einheit.set_selected(0)
    row("Menge:", [window.rech_menge, window.rech_einheit])

    window.rech_preis_netto = Gtk.Entry()
    window.rech_preis_netto.set_text("0,00")
    window.rech_preis_netto.set_hexpand(True)
    window.rech_steuer = Gtk.DropDown.new_from_strings(["7%", "19%"])
    window.rech_steuer.set_selected(1)
    price_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
    price_box.append(window.rech_preis_netto)
    price_box.append(Gtk.Label(label="€ Netto"))
    price_box.append(window.rech_steuer)
    price_box.append(Gtk.Label(label="Steuer"))
    row("Preis & Steuer:", price_box)

    # Zahlungsziel (Einzel)
    window.rech_ziel = Gtk.Entry()
    window.rech_ziel.set_placeholder_text("z. B. 14 Tage netto oder 2025-12-31")
    row("Zahlungsziel:", window.rech_ziel)

    # Artikel/Steuer ändern → VK-Brutto aus Stammdaten übernehmen → Netto setzen
    window.rech_artikel_dd.connect("notify::selected", partial(on_artikel_changed_set_brutto, window))
    window.rech_steuer.connect("notify::selected", partial(on_artikel_changed_set_brutto, window))

    btn_single = Gtk.Button(label="Rechnung (Einzelposition) erstellen")
    btn_single.add_css_class("suggested-action")
    btn_single.connect("clicked", partial(on_rechnung_save_single, window))
    form.append(btn_single)

    frame_single.add(form)
    page.append(frame_single)

    # ---------- Mehrere Verkäufe ----------
    frame_multi = Adw.PreferencesGroup(title="Rechnung aus mehreren Verkäufen")
    box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=12)

    # Filter (Kunde + Zeitraum)
    filter_row = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
    kunden_filter = ["Alle Kunden"] + kunden_namen
    window.rech_sel_kunde = Gtk.DropDown.new_from_strings(kunden_filter)
    window.rech_sel_kunde.connect("notify::selected", lambda *_: refresh_offene_verkaeufe_list(window))
    filter_row.append(Gtk.Label(label="Kunde:"))
    filter_row.append(window.rech_sel_kunde)

    monate = ["Alle Monate"] + [f"{i:02d}" for i in range(1, 13)]
    window.rech_sel_monat = Gtk.DropDown.new_from_strings(monate)
    window.rech_sel_monat.connect("notify::selected", lambda *_: refresh_offene_verkaeufe_list(window))
    filter_row.append(Gtk.Label(label="Monat:"))
    filter_row.append(window.rech_sel_monat)

    jahre = set()
    for v in window.app.data.get("verkäufe", []):
        d = str(v.get("datum", ""))[:4]
        if d:
            jahre.add(d)
    window.rech_sel_jahr = Gtk.DropDown.new_from_strings(["Alle Jahre"] + sorted(jahre, reverse=True))
    window.rech_sel_jahr.connect("notify::selected", lambda *_: refresh_offene_verkaeufe_list(window))
    filter_row.append(Gtk.Label(label="Jahr:"))
    filter_row.append(window.rech_sel_jahr)
    box.append(filter_row)

    # Zahlungsziel (Mehrfach)
    ziel_row = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
    ziel_row.append(Gtk.Label(label="Zahlungsziel:"))
    window.rech_sel_ziel = Gtk.Entry()
    window.rech_sel_ziel.set_placeholder_text("z. B. 14 Tage netto oder 2025-12-31")
    ziel_row.append(window.rech_sel_ziel)
    box.append(ziel_row)

    # Liste offener Verkäufe
    sc = Gtk.ScrolledWindow()
    sc.set_vexpand(True)
    sc.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.AUTOMATIC)
    window.rech_sel_list = Gtk.ListBox()
    window.rech_sel_list.add_css_class("boxed-list")
    sc.set_child(window.rech_sel_list)
    box.append(sc)

    btn_create = Gtk.Button(label="Rechnung aus Auswahl erstellen")
    btn_create.add_css_class("suggested-action")
    btn_create.connect("clicked", partial(on_rechnung_save_from_selection, window))
    box.append(btn_create)

    frame_multi.add(box)
    page.append(frame_multi)

    window.content_stack.add_named(page, "rechnung")

    # initial füllen
    refresh_offene_verkaeufe_list(window)


def on_artikel_changed_set_brutto(window, *_):
    """VK-Brutto aus Stammdaten übernehmen und Netto berechnen."""
    try:
        idx_art = window.rech_artikel_dd.get_selected()
        if idx_art == Gtk.INVALID_LIST_POSITION:
            return
        artikel_name = window.rech_artikel_dd.get_model().get_string(idx_art)
        artikel = next((a for a in window.app.data["stammdaten"]["artikel"] if a["name"] == artikel_name), None)
        if not artikel:
            return
        brutto = float(artikel.get("vk_preis", 0) or 0)
        steuer = 0.07 if window.rech_steuer.get_selected() == 0 else 0.19
        netto = brutto / (1 + steuer)
        window.rech_preis_netto.set_text(f"{netto:.4f}".replace(".", ","))
    except Exception:
        pass


def on_rechnung_save_single(window, _btn):
    """Einzelrechnung speichern"""
    try:
        datum = window.rech_datum.get_text().strip()
        idx_kunde = window.rech_kunde_dd.get_selected()
        if idx_kunde == Gtk.INVALID_LIST_POSITION:
            show_message(window, "Fehler", "Bitte einen Kunden auswählen.")
            return
        kunde = window.rech_kunde_dd.get_model().get_string(idx_kunde)

        idx_art = window.rech_artikel_dd.get_selected()
        if idx_art == Gtk.INVALID_LIST_POSITION:
            show_message(window, "Fehler", "Bitte Artikel auswählen.")
            return
        artikel = window.rech_artikel_dd.get_model().get_string(idx_art)

        menge = float(window.rech_menge.get_value())
        einheit = EINHEITEN[window.rech_einheit.get_selected()]
        preis = float(window.rech_preis_netto.get_text().replace(",", "."))
        steuer = 7 if window.rech_steuer.get_selected() == 0 else 19
        gesamt = round(menge * preis * (1 + steuer / 100), 2)
        ziel = window.rech_ziel.get_text().strip()

        nr = next_rechnung_nr(window.app.conn)
        window.app.data["verkäufe"].append({
            "datum": datum,
            "artikel": artikel,
            "menge": menge,
            "einheit": einheit,
            "preis": preis,
            "steuer": steuer,
            "gesamt": gesamt,
            "kunde": kunde,
            "rechnung_nr": nr,
            "zahlungsziel": ziel,
        })
        window.app.save_data()
        show_message(window, "Erfolg", f"Rechnung {nr} gespeichert.")

        # Offene Posten + Rechnungsübersicht aktualisieren
        refresh_offene_verkaeufe_list(window)
        update_rechnung_jahr_filter(window)
        refresh_rechnung_list(window)

    except Exception as e:
        show_message(window, "Fehler", str(e))


# ==========================================
# Rechnung aus mehreren Verkäufen
# ==========================================

def refresh_offene_verkaeufe_list(window):
    if not hasattr(window, "rech_sel_list"):
        return
    while row := window.rech_sel_list.get_row_at_index(0):
        window.rech_sel_list.remove(row)

    window._rech_sel_checks = []
    kunde_idx = window.rech_sel_kunde.get_selected()
    kunde = window.rech_sel_kunde.get_model().get_string(kunde_idx) if kunde_idx > 0 else None
    monat_idx = window.rech_sel_monat.get_selected()
    monat = f"{monat_idx:02d}" if monat_idx > 0 else None
    jahr_idx = window.rech_sel_jahr.get_selected()
    jahr_model = window.rech_sel_jahr.get_model()
    jahr = jahr_model.get_string(jahr_idx) if jahr_idx > 0 else None

    for v in reversed(window.app.data.get("verkäufe", [])):
        if v.get("rechnung_nr"):
            continue
        if kunde and v.get("kunde") != kunde:
            continue
        if jahr and not str(v.get("datum", "")).startswith(jahr):
            continue
        if monat and str(v.get("datum", ""))[5:7] != monat:
            continue

        row = Gtk.ListBoxRow()
        box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
        box.set_margin_start(10)
        box.set_margin_end(10)
        box.set_margin_top(5)
        box.set_margin_bottom(5)

        chk = Gtk.CheckButton()
        box.append(chk)

        txt = Gtk.Label(
            label=f"{v.get('datum')} – {v.get('kunde')} – {v.get('artikel')} ({v.get('menge')} {v.get('einheit')})"
        )
        txt.set_halign(Gtk.Align.START)
        txt.set_hexpand(True)
        box.append(txt)

        try:
            brutto = float(v["menge"]) * float(v["preis"]) * (1 + (v["steuer"] / 100))
        except Exception:
            brutto = 0.0
        sum_lbl = Gtk.Label(label=f"{brutto:.2f} €")
        sum_lbl.add_css_class("dim-label")
        box.append(sum_lbl)

        row.set_child(box)
        window.rech_sel_list.append(row)
        window._rech_sel_checks.append((v, chk))


def on_rechnung_save_from_selection(window, _btn):
    selected = [v for v, chk in getattr(window, "_rech_sel_checks", []) if chk.get_active()]
    if not selected:
        show_message(window, "Hinweis", "Keine Verkäufe ausgewählt.")
        return

    kunden = {v.get("kunde") for v in selected}
    if len(kunden) > 1:
        show_message(window, "Fehler", "Bitte nur Verkäufe eines Kunden auswählen.")
        return

    ziel = window.rech_sel_ziel.get_text().strip()
    nr = next_rechnung_nr(window.app.conn)
    for v in selected:
        v["rechnung_nr"] = nr
        v["zahlungsziel"] = ziel
    window.app.save_data()
    show_message(window, "Erfolg", f"Rechnung {nr} mit {len(selected)} Positionen erstellt.")

    # Listen aktualisieren
    refresh_offene_verkaeufe_list(window)
    update_rechnung_jahr_filter(window)
    refresh_rechnung_list(window)


# ==========================================
# Rechnungs-Übersicht (gruppiert) + Filter
# ==========================================

def create_rechnung_list_page(window):
    page = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=20)
    page.set_margin_start(40)
    page.set_margin_end(40)
    page.set_margin_top(40)
    page.set_margin_bottom(40)

    header = Gtk.Label(label="Rechnungen")
    header.add_css_class("title-2")
    header.set_halign(Gtk.Align.START)
    page.append(header)

    # Filterleiste
    filters = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=10)

    # Suchfeld
    box_search = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
    box_search.append(Gtk.Label(label="Suchen:"))
    window.rechnung_filter = Gtk.Entry()
    window.rechnung_filter.set_placeholder_text("Rechnungsnr., Kunde …")
    window.rechnung_filter.connect("changed", lambda e: refresh_rechnung_list(window))
    box_search.append(window.rechnung_filter)
    filters.append(box_search)

    # Datum-Filter
    box_date = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
    box_date.append(Gtk.Label(label="Datum:"))
    window.rechnung_filter_jahr = Gtk.DropDown()
    window.rechnung_filter_jahr.connect("notify::selected", lambda d, p: refresh_rechnung_list(window))

    monate = ["Alle Monate"] + [f"{i:02d}" for i in range(1, 13)]
    window.rechnung_filter_monat = Gtk.DropDown.new_from_strings(monate)
    window.rechnung_filter_monat.connect("notify::selected", lambda d, p: refresh_rechnung_list(window))

    box_date.append(window.rechnung_filter_jahr)
    box_date.append(window.rechnung_filter_monat)
    filters.append(box_date)

    page.append(filters)

    # Liste
    scrolled = Gtk.ScrolledWindow()
    scrolled.set_vexpand(True)
    scrolled.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.AUTOMATIC)
    window.rechnung_list_box = Gtk.ListBox()
    window.rechnung_list_box.add_css_class("boxed-list")
    scrolled.set_child(window.rechnung_list_box)
    page.append(scrolled)

    window.content_stack.add_named(page, "rechnung_list")

    update_rechnung_jahr_filter(window)
    refresh_rechnung_list(window)


def update_rechnung_jahr_filter(window):
    # Jahr aus allen Verkäufen, die eine Rechnungsnummer besitzen
    jahre = {"Alle Jahre"}
    for v in window.app.data.get("verkäufe", []):
        if not v.get("rechnung_nr"):
            continue
        try:
            jahre.add(str(v.get("datum", ""))[:4])
        except Exception:
            pass

    jahre_liste = sorted(list(jahre), reverse=True)
    if "Alle Jahre" in jahre_liste:
        jahre_liste.remove("Alle Jahre")
        jahre_liste.insert(0, "Alle Jahre")

    selected_index = window.rechnung_filter_jahr.get_selected() if hasattr(window, "rechnung_filter_jahr") else 0
    selected_value = None
    if selected_index > 0 and window.rechnung_filter_jahr.get_model():
        selected_value = window.rechnung_filter_jahr.get_model().get_string(selected_index)

    window.rechnung_filter_jahr.set_model(Gtk.StringList.new(jahre_liste))

    if selected_value and selected_value in jahre_liste:
        window.rechnung_filter_jahr.set_selected(jahre_liste.index(selected_value))
    else:
        window.rechnung_filter_jahr.set_selected(0)


def refresh_rechnung_list(window):
    if not hasattr(window, "rechnung_list_box"):
        return

    # Liste leeren
    while window.rechnung_list_box.get_row_at_index(0) is not None:
        window.rechnung_list_box.remove(window.rechnung_list_box.get_row_at_index(0))

    # Filter lesen
    q = window.rechnung_filter.get_text().lower() if hasattr(window, "rechnung_filter") else ""
    model_jahr = window.rechnung_filter_jahr.get_model() if hasattr(window, "rechnung_filter_jahr") else None
    idx_jahr = window.rechnung_filter_jahr.get_selected() if hasattr(window, "rechnung_filter_jahr") else 0
    filter_jahr = model_jahr.get_string(idx_jahr) if model_jahr and idx_jahr != Gtk.INVALID_LIST_POSITION else "Alle Jahre"
    idx_monat = window.rechnung_filter_monat.get_selected() if hasattr(window, "rechnung_filter_monat") else 0
    filter_monat = f"{idx_monat:02d}" if idx_monat > 0 else "Alle Monate"

    # Gruppieren nach Rechnungsnummer
    invoices = {}
    for v in window.app.data.get("verkäufe", []):
        nr = v.get("rechnung_nr")
        if not nr:
            continue
        invoices.setdefault(nr, []).append(v)

    # Anzeigen (neueste zuerst anhand max Datum)
    for nr, items in sorted(invoices.items(), key=lambda kv: max(str(i.get("datum","")) for i in kv[1]), reverse=True):
        datum = sorted([str(i.get("datum","")) for i in items])[0] if items else ""
        if filter_jahr != "Alle Jahre" and not datum.startswith(filter_jahr):
            continue
        if filter_monat != "Alle Monate" and datum[5:7] != filter_monat:
            continue

        kunde_set = {i.get("kunde","") for i in items}
        kunde = next(iter(kunde_set)) if kunde_set else ""
        ziel = next((i.get("zahlungsziel") for i in items if i.get("zahlungsziel")), "") or ""

        hay = f"{nr} {kunde} {ziel}".lower()
        if q and q not in hay:
            continue

        # Summe brutto
        brutto_sum = 0.0
        for i in items:
            try:
                menge = float(i.get("menge", 0) or 0.0)
                preis = float(i.get("preis", 0) or 0.0)     # Netto
                st = int(i.get("steuer", 0) or 0)
                brutto_sum += menge * preis * (1 + st / 100.0)
            except Exception:
                pass

        row = Gtk.ListBoxRow()
        outer = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=6)
        outer.set_margin_start(10); outer.set_margin_end(10); outer.set_margin_top(10); outer.set_margin_bottom(10)

        top = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
        title = Gtk.Label(label=f"Rechnung {nr} – {kunde} ({len(items)} Positionen)")
        title.add_css_class("heading")
        title.set_halign(Gtk.Align.START)
        title.set_hexpand(True)
        top.append(title)

        sum_lbl = Gtk.Label(label=f"{brutto_sum:.2f} €")
        sum_lbl.add_css_class("title-3")
        top.append(sum_lbl)
        outer.append(top)

        ziel_txt = f" | Zahlungsziel: {ziel}" if ziel else ""
        details = Gtk.Label(label=f"{datum}{ziel_txt}")
        details.add_css_class("dim-label")
        details.set_halign(Gtk.Align.START)
        outer.append(details)

        # Aktionen: PDF + ZUGFeRD + Bearbeiten + Löschen
        btn_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)

        btn_pdf = Gtk.Button(label="PDF")
        btn_pdf.add_css_class("flat")
        btn_pdf.connect("clicked", partial(export_rechnung_pdf, window, nr))
        btn_box.append(btn_pdf)

        btn_zugferd = Gtk.Button(label="ZUGFeRD-PDF")
        btn_zugferd.add_css_class("flat")
        btn_zugferd.connect("clicked", partial(export_rechnung_zugferd, window, nr))
        btn_box.append(btn_zugferd)

        btn_edit = Gtk.Button(label="Bearbeiten")
        btn_edit.add_css_class("flat")
        btn_edit.connect("clicked", partial(open_rechnung_editor, window, nr))
        btn_box.append(btn_edit)

        btn_del = Gtk.Button(label="Löschen")
        btn_del.add_css_class("destructive-action")
        btn_del.add_css_class("flat")
        btn_del.connect("clicked", partial(delete_rechnung, window, nr))
        btn_box.append(btn_del)

        outer.append(btn_box)

        row.set_child(outer)
        window.rechnung_list_box.append(row)


# ==========================================
# Bearbeiten & Löschen
# ==========================================

def open_rechnung_editor(window, rechnung_nr, _btn=None):
    """Rechnungs-Metadaten (Datum, Kunde, Zahlungsziel) anpassen."""
    items = [v for v in window.app.data.get("verkäufe", []) if v.get("rechnung_nr") == rechnung_nr]
    if not items:
        show_message(window, "Fehler", "Für diese Rechnungsnummer wurden keine Positionen gefunden.")
        return

    # aktuelle Werte ermitteln
    current_date = sorted([str(i.get("datum","")) for i in items if i.get("datum")])[0] if items else ""
    current_customer = next((i.get("kunde") for i in items if i.get("kunde")), "")
    current_ziel = next((i.get("zahlungsziel") for i in items if i.get("zahlungsziel")), "") or ""

    dlg = Adw.MessageDialog.new(window, f"Rechnung {rechnung_nr} bearbeiten", "Passen Sie die Daten an.")
    dlg.add_response("cancel", "Abbrechen")
    dlg.add_response("save", "Speichern")
    dlg.set_response_appearance("save", Adw.ResponseAppearance.SUGGESTED)

    content = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=12)
    content.set_margin_start(16); content.set_margin_end(16)
    content.set_margin_top(16); content.set_margin_bottom(16)

    def row(lbl, widget):
        hb = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
        l = Gtk.Label(label=lbl); l.set_size_request(150, -1); l.set_halign(Gtk.Align.START)
        hb.append(l)
        if isinstance(widget, list):
            for w in widget: hb.append(w)
        else:
            hb.append(widget)
        content.append(hb)

    # Felder
    e_datum = Gtk.Entry(); e_datum.set_text(current_date or "")
    row("Datum:", e_datum)

    kunden_namen = [k["name"] for k in window.app.data["stammdaten"]["kunden"]]
    dd_kunde = Gtk.DropDown(); dd_kunde.set_model(Gtk.StringList.new(kunden_namen)); dd_kunde.set_enable_search(True)
    try:
        dd_kunde.set_selected(kunden_namen.index(current_customer))
    except ValueError:
        dd_kunde.set_selected(Gtk.INVALID_LIST_POSITION)
    row("Kunde:", dd_kunde)

    e_ziel = Gtk.Entry(); e_ziel.set_placeholder_text("z. B. 14 Tage netto oder 2025-12-31")
    e_ziel.set_text(current_ziel)
    row("Zahlungsziel:", e_ziel)

    dlg.set_extra_child(content)

    def on_resp(_d, resp):
        if resp != "save":
            return
        new_date = e_datum.get_text().strip()
        idx_k = dd_kunde.get_selected()
        new_customer = dd_kunde.get_model().get_string(idx_k) if idx_k != Gtk.INVALID_LIST_POSITION else current_customer
        new_ziel = e_ziel.get_text().strip()

        for i in items:
            if new_date:
                i["datum"] = new_date
            if new_customer:
                i["kunde"] = new_customer
            i["zahlungsziel"] = new_ziel

        window.app.save_data()
        update_rechnung_jahr_filter(window)
        refresh_rechnung_list(window)
        try:
            refresh_offene_verkaeufe_list(window)
        except Exception:
            pass

    dlg.connect("response", on_resp)
    dlg.show()


def delete_rechnung(window, rechnung_nr, _btn=None):
    """Rechnung ENDTGÜLTIG löschen = alle zugehörigen Verkäufe entfernen."""
    if not confirm(window, "Rechnung löschen?",
                   f"Sollen wirklich ALLE Verkäufe der Rechnung {rechnung_nr} endgültig entfernt werden?\n"
                   "Dieser Vorgang kann nicht rückgängig gemacht werden."):
        return

    before = len(window.app.data.get("verkäufe", []))
    window.app.data["verkäufe"] = [v for v in window.app.data.get("verkäufe", []) if v.get("rechnung_nr") != rechnung_nr]
    after = len(window.app.data.get("verkäufe", []))

    if before == after:
        show_message(window, "Hinweis", "Es wurden keine Positionen gefunden.")
        return

    window.app.save_data()
    # Ansichten aktualisieren
    update_rechnung_jahr_filter(window)
    refresh_rechnung_list(window)
    try:
        refresh_offene_verkaeufe_list(window)
    except Exception:
        pass
    show_message(window, "Erfolg", f"Rechnung {rechnung_nr} und alle zugehörigen Verkäufe wurden entfernt.")


# ==========================================
# PDF-Erzeugung mit Logo & Zahlungsziel
# ==========================================

def export_rechnung_pdf(window, rechnung_nr, _btn=None):
    if not REPORTLAB_OK:
        show_message(window, "PDF nicht möglich", "Installiere reportlab: pip3 install reportlab")
        return

    try:
        base = Path(__file__).resolve().parent
        logo_path = base / "icons" / "logo.png"
        out_dir = base / "rechnungen"
        out_dir.mkdir(exist_ok=True)
        out_file = out_dir / f"Rechnung_{rechnung_nr}.pdf"

        items = [v for v in window.app.data["verkäufe"] if v.get("rechnung_nr") == rechnung_nr]
        if not items:
            show_message(window, "Fehler", "Keine Positionen gefunden.")
            return

        # --- Minimaler PDF-Aufbau (Platzhalter; dein bestehender Aufbau kann hier bleiben) ---
        c = canvas.Canvas(str(out_file), pagesize=A4)
        w, h = A4
        y = h - 30 * mm
        c.setFont("Helvetica-Bold", 14)
        c.drawString(20 * mm, y, f"Rechnung {rechnung_nr}")
        y -= 10 * mm
        c.setFont("Helvetica", 10)
        kunde = next((i.get("kunde", "") for i in items if i.get("kunde")), "")
        ziel = next((i.get("zahlungsziel", "") for i in items if i.get("zahlungsziel")), "")
        c.drawString(20 * mm, y, f"Kunde: {kunde}")
        y -= 6 * mm
        if ziel:
            c.drawString(20 * mm, y, f"Zahlungsziel: {ziel}")
            y -= 6 * mm
        y -= 4 * mm
        total = 0.0
        for i in items:
            m = float(i.get("menge", 0))
            p = float(i.get("preis", 0))
            st = int(i.get("steuer", 0))
            brutto = m * p * (1 + st / 100.0)
            total += brutto
            c.drawString(20 * mm, y, f"{i.get('datum','')} | {i.get('artikel','')} x {m} {i.get('einheit','')} – {brutto:.2f} €")
            y -= 5 * mm
        y -= 5 * mm
        c.setFont("Helvetica-Bold", 12)
        c.drawString(20 * mm, y, f"Summe: {total:.2f} €")
        c.showPage()
        c.save()

        show_message(window, "PDF gespeichert", str(out_file))
    except Exception as e:
        show_message(window, "Fehler", str(e))
