from utils import Gtk, Adw, partial, show_message
from datetime import datetime
from pathlib import Path

# ReportLab (für PDF)
try:
    from reportlab.lib.pagesizes import A4
    from reportlab.lib import colors
    from reportlab.lib.units import mm
    from reportlab.platypus import SimpleDocTemplate, Table, TableStyle, Paragraph, Spacer
    from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
    from reportlab.pdfgen.canvas import Canvas
    REPORTLAB_AVAILABLE = True
except Exception:
    REPORTLAB_AVAILABLE = False


# ----------------- Helfer -----------------
def _fmt_date_iso_to_de(iso_date: str) -> str:
    try:
        return datetime.strptime(iso_date, "%Y-%m-%d").strftime("%d.%m.%y")
    except Exception:
        return iso_date or ""


def _calc_brutto_from_einkauf(e):
    try:
        if e.get("gesamt_brutto") not in (None, ""):
            return float(e["gesamt_brutto"])
        menge = float(e.get("menge", 0) or 0)
        pnetto = float(e.get("preis_netto", 0) or 0)
        st = float(e.get("steuer", 0) or 0)
        return round(menge * pnetto * (1 + st / 100.0), 2)
    except Exception:
        return 0.0


def _calc_brutto_from_verkauf(v):
    try:
        if v.get("gesamt") not in (None, ""):
            return float(v["gesamt"])
        menge = float(v.get("menge", 0) or 0)
        pnetto = float(v.get("preis", 0) or 0)
        st = float(v.get("steuer", 0) or 0)
        return round(menge * pnetto * (1 + st / 100.0), 2)
    except Exception:
        return 0.0


def _fmt_euro(x: float) -> str:
    return f"{x:,.2f} €".replace(",", "X").replace(".", ",").replace("X", ".")


# ----------------- GUI-Seite -----------------
def create_kassenbuch_page(window):
    page = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=16)
    page.set_margin_start(24)
    page.set_margin_end(24)
    page.set_margin_top(24)
    page.set_margin_bottom(24)

    # Kopfzeile mit Titel, Buttons
    header_row = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
    title = Gtk.Label(label="Kassenbuch")
    title.add_css_class("title-2")
    title.set_halign(Gtk.Align.START)
    title.set_hexpand(True)
    header_row.append(title)

    btn_set_anfang = Gtk.Button(label="Anfangsbestand setzen")
    btn_set_anfang.connect("clicked", partial(on_set_anfangsbestand, window))
    header_row.append(btn_set_anfang)

    btn_privat = Gtk.Button(label="Privatentnahme")
    btn_privat.connect("clicked", partial(on_add_privatentnahme, window))
    header_row.append(btn_privat)

    page.append(header_row)

    # Monat/Jahr Auswahl
    filter_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=8)
    filter_box.append(Gtk.Label(label="Zeitraum:"))

    window.kb_month = Gtk.DropDown.new_from_strings([f"{i:02d}" for i in range(1, 13)])
    window.kb_month.set_selected(datetime.now().month - 1)
    window.kb_month.connect("notify::selected", lambda *_: refresh_kassenbuch(window))
    filter_box.append(window.kb_month)

    years = set()
    for dsrc in ("einkäufe", "verkäufe", "privatentnahmen"):
        for row in window.app.data.get(dsrc, []):
            d = row.get("datum")
            if d:
                years.add(d[:4])
    if not years:
        years.add(str(datetime.now().year))
    years_list = sorted(years)
    window.kb_year = Gtk.DropDown.new_from_strings(years_list)
    try:
        sel_idx = years_list.index(str(datetime.now().year))
    except ValueError:
        sel_idx = 0
    window.kb_year.set_selected(sel_idx)
    window.kb_year.connect("notify::selected", lambda *_: refresh_kassenbuch(window))
    filter_box.append(window.kb_year)
    page.append(filter_box)
    
    # Einnahmen-Filter
    einnahmen_filter_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=8)
    einnahmen_filter_box.append(Gtk.Label(label="Einnahmen anzeigen:"))
    
    einnahmen_modi = ["Alle Verkäufe", "Nur Barverkäufe (ohne Rechnung)", "Nur Rechnungen"]
    window.kb_einnahmen_filter = Gtk.DropDown.new_from_strings(einnahmen_modi)
    window.kb_einnahmen_filter.set_selected(0)
    window.kb_einnahmen_filter.connect("notify::selected", lambda *_: refresh_kassenbuch(window))
    einnahmen_filter_box.append(window.kb_einnahmen_filter)
    page.append(einnahmen_filter_box)

    # --- Info-Karten (breiter) ---
    cards_row = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=12)
    cards_row.set_homogeneous(True)

    def make_card(title_text):
        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=6)
        box.add_css_class("card")
        box.add_css_class("boxed-list")
        box.set_hexpand(True)
        box.set_size_request(260, -1)
        title = Gtk.Label(label=title_text)
        title.set_halign(Gtk.Align.CENTER)
        title.add_css_class("dim-label")
        value = Gtk.Label(label="0,00 €")
        value.add_css_class("title-3")
        value.set_halign(Gtk.Align.CENTER)
        box.append(title)
        box.append(value)
        return box, value

    card_ab, window.kb_lbl_ab = make_card("Anfangsbestand")
    card_ein, window.kb_lbl_ein = make_card("Einnahmen")
    card_aus, window.kb_lbl_aus = make_card("Ausgaben")
    card_pri, window.kb_lbl_pri = make_card("Privatentnahmen")
    
    # *** NEU: Farben für Einnahmen (grün) und Ausgaben (rot) setzen ***
    window.kb_lbl_ein.add_css_class("success")
    window.kb_lbl_aus.add_css_class("error")

    cards_row.append(card_ab)
    cards_row.append(card_ein)
    cards_row.append(card_aus)
    cards_row.append(card_pri)
    page.append(cards_row)

    # Aktueller Kassenbestand
    saldo_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=6)
    saldo_box.add_css_class("card")
    saldo_box.add_css_class("boxed-list")
    saldo_box.set_margin_top(6)
    saldo_title = Gtk.Label(label="Aktueller Kassenbestand")
    saldo_title.add_css_class("title-3")
    saldo_title.set_halign(Gtk.Align.CENTER)
    window.kb_lbl_saldo = Gtk.Label(label="0,00 €")
    window.kb_lbl_saldo.add_css_class("title-2")
    window.kb_lbl_saldo.set_halign(Gtk.Align.CENTER)
    saldo_box.append(saldo_title)
    saldo_box.append(window.kb_lbl_saldo)
    page.append(saldo_box)

    # --- Zwei Spalten nebeneinander ---
    twin = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=12)
    twin.set_vexpand(True)

    def build_side(title_text):
        side = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=6)
        h = Gtk.Label(label=title_text)
        h.add_css_class("title-3")
        h.set_halign(Gtk.Align.CENTER)
        side.append(h)

        header = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=6)
        for cap in ("Nr.:", "Datum", "Beleg", "Betrag"):
            lab = Gtk.Label(label=cap)
            lab.add_css_class("heading")
            lab.set_halign(Gtk.Align.START)
            if cap == "Betrag":
                lab.set_halign(Gtk.Align.END)
            if cap == "Beleg":
                lab.set_hexpand(True)
            header.append(lab)
        side.append(header)

        sc = Gtk.ScrolledWindow()
        sc.set_vexpand(True)
        lst = Gtk.ListBox()
        lst.add_css_class("boxed-list")
        sc.set_child(lst)
        side.append(sc)
        return side, lst

    left_box, window.kb_left_list = build_side("Ausgaben")
    right_box, window.kb_right_list = build_side("Einnahmen")
    twin.append(left_box)
    twin.append(right_box)
    page.append(twin)

    # PDF-Button
    btn_pdf = Gtk.Button(label="PDF exportieren")
    btn_pdf.connect("clicked", partial(on_export_pdf, window))
    if not REPORTLAB_AVAILABLE:
        btn_pdf.set_sensitive(False)
        btn_pdf.set_tooltip_text("ReportLab nicht installiert (pip install reportlab)")
    page.append(btn_pdf)

    window.content_stack.add_named(page, "kassenbuch")
    refresh_kassenbuch(window)


def refresh_kassenbuch(window):
    if not hasattr(window, "kb_month"):
        return
    month = f"{window.kb_month.get_selected() + 1:02d}"
    year_model = window.kb_year.get_model()
    year = year_model.get_string(window.kb_year.get_selected()) if year_model else str(datetime.now().year)

    einnahmen_filter_mode = window.kb_einnahmen_filter.get_selected()

    def clear(lb):
        while True:
            row = lb.get_row_at_index(0)
            if not row:
                break
            lb.remove(row)

    clear(window.kb_left_list)
    clear(window.kb_right_list)

    anfang = float(window.app.data.get("anfangsbestand", 0.0))

    ausgaben = []
    privat = []
    for e in window.app.data.get("einkäufe", []):
        d = str(e.get("datum", ""))
        if not d.startswith(f"{year}-{month}"):
            continue
        betrag = _calc_brutto_from_einkauf(e)
        beleg = e.get("rechnungs_nr") and f"Rechnung {e.get('rechnungs_nr')}" or (e.get("lieferant") or "Einkauf")
        ausgaben.append((d, beleg, betrag))

    for p in window.app.data.get("privatentnahmen", []):
        d = str(p.get("datum", ""))
        if not d.startswith(f"{year}-{month}"):
            continue
        betrag = float(p.get("betrag", 0) or 0)
        privat.append((d, "Privatentnahme", betrag))
        ausgaben.append((d, "Privatentnahme", betrag))

    einnahmen = []
    for v in window.app.data.get("verkäufe", []):
        d = str(v.get("datum", ""))
        if not d.startswith(f"{year}-{month}"):
            continue
            
        hat_rechnung = bool(v.get("rechnung_nr"))
        if einnahmen_filter_mode == 1 and hat_rechnung:
            continue
        if einnahmen_filter_mode == 2 and not hat_rechnung:
            continue

        betrag = _calc_brutto_from_verkauf(v)
        beleg = v.get("rechnung_nr") and f"Rechnung {v.get('rechnung_nr')}" or "Verkaufsliste"
        einnahmen.append((d, beleg, betrag))

    ausgaben.sort(key=lambda x: x[0])
    einnahmen.sort(key=lambda x: x[0])

    sum_ein = sum(b for _, _, b in einnahmen)
    sum_aus = sum(b for _, _, b in ausgaben)
    sum_pri = sum(b for _, _, b in privat)
    saldo = anfang + sum_ein - sum_aus

    window.kb_lbl_ab.set_label(_fmt_euro(anfang))
    window.kb_lbl_ein.set_label(_fmt_euro(sum_ein))
    window.kb_lbl_aus.set_label(_fmt_euro(sum_aus))
    window.kb_lbl_pri.set_label(_fmt_euro(sum_pri))
    window.kb_lbl_saldo.set_label(_fmt_euro(saldo))
    
    # *** NEU: Farbe für Saldo basierend auf Wert setzen ***
    window.kb_lbl_saldo.remove_css_class("success")
    window.kb_lbl_saldo.remove_css_class("error")
    if saldo >= 0:
        window.kb_lbl_saldo.add_css_class("success")
    else:
        window.kb_lbl_saldo.add_css_class("error")

    def add_row(listbox, idx, datum, beleg, betrag):
        row = Gtk.ListBoxRow()
        hb = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=6)
        hb.set_margin_start(6)
        hb.set_margin_end(6)
        hb.set_margin_top(4)
        hb.set_margin_bottom(4)
        lidx = Gtk.Label(label=str(idx))
        lidx.set_halign(Gtk.Align.START)
        lidx.set_size_request(40, -1)
        ldat = Gtk.Label(label=_fmt_date_iso_to_de(datum))
        ldat.set_halign(Gtk.Align.START)
        ldat.set_size_request(90, -1)
        lbel = Gtk.Label(label=str(beleg))
        lbel.set_halign(Gtk.Align.START)
        lbel.set_hexpand(True)
        lbet = Gtk.Label(label=_fmt_euro(betrag))
        lbet.set_halign(Gtk.Align.END)
        lbet.set_size_request(100, -1)
        hb.append(lidx)
        hb.append(ldat)
        hb.append(lbel)
        hb.append(lbet)
        row.set_child(hb)
        listbox.append(row)

    for i, (d, beleg, betrag) in enumerate(ausgaben, start=1):
        add_row(window.kb_left_list, i, d, beleg, betrag)
    for i, (d, beleg, betrag) in enumerate(einnahmen, start=1):
        add_row(window.kb_right_list, i, d, beleg, betrag)


# ----------------- Dialoge -----------------
def on_set_anfangsbestand(window, _btn=None):
    dlg = Adw.MessageDialog.new(window, "Anfangsbestand setzen", "Bitte neuen Anfangsbestand eingeben.")
    box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=8)
    spin = Gtk.SpinButton()
    spin.set_adjustment(Gtk.Adjustment(value=float(window.app.data.get("anfangsbestand", 0.0)),
                                       lower=-1_000_000, upper=1_000_000, step_increment=1, page_increment=10))
    spin.set_digits(2)
    box.append(spin)
    dlg.set_extra_child(box)
    dlg.add_response("cancel", "Abbrechen")
    dlg.add_response("ok", "Speichern")
    dlg.set_response_appearance("ok", Adw.ResponseAppearance.SUGGESTED)
    dlg.connect("response", lambda d, r: _save_anfangsbestand(window, spin) if r == "ok" else None)
    dlg.present()


def _save_anfangsbestand(window, spin):
    try:
        window.app.data["anfangsbestand"] = float(spin.get_value())
        window.app.save_data()
        refresh_kassenbuch(window)
        show_message(window, "Gespeichert", "Anfangsbestand wurde aktualisiert.")
    except Exception as e:
        show_message(window, "Fehler", f"{e}")


def on_add_privatentnahme(window, _btn=None):
    dlg = Adw.MessageDialog.new(window, "Privatentnahme", "Datum und Betrag eingeben.")
    vbox = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=8)

    entry_date = Gtk.Entry()
    entry_date.set_text(datetime.now().strftime("%Y-%m-%d"))
    entry_amount = Gtk.Entry()
    entry_amount.set_placeholder_text("Betrag in € (z. B. 50,00)")

    vbox.append(Gtk.Label(label="Datum (YYYY-MM-DD):"))
    vbox.append(entry_date)
    vbox.append(Gtk.Label(label="Betrag:"))
    vbox.append(entry_amount)

    dlg.set_extra_child(vbox)
    dlg.add_response("cancel", "Abbrechen")
    dlg.add_response("ok", "Hinzufügen")
    dlg.set_response_appearance("ok", Adw.ResponseAppearance.SUGGESTED)

    def on_resp(d, response):
        if response != "ok":
            return
        try:
            datum = entry_date.get_text().strip()
            datetime.strptime(datum, "%Y-%m-%d")
            betrag = float(entry_amount.get_text().replace(",", "."))
            window.app.data.setdefault("privatentnahmen", []).append({"datum": datum, "betrag": betrag})
            window.app.save_data()
            refresh_kassenbuch(window)
            show_message(window, "Erfolgreich", "Privatentnahme erfasst.")
        except Exception:
            show_message(window, "Fehler", "Bitte gültiges Datum und Betrag angeben.")

    dlg.connect("response", on_resp)
    dlg.present()


# ----------------- PDF-Export (Kopf, 4 Kennzahlen, 2 Spalten, Fuß) -----------------
class NumberedCanvas(Canvas):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._saved_page_states = []

    def showPage(self):
        self._saved_page_states.append(dict(self.__dict__))
        self._startPage()

    def save(self):
        page_count = len(self._saved_page_states)
        for i, state in enumerate(self._saved_page_states):
            self.__dict__.update(state)
            self._pageNumber = i + 1
            self._draw_footer(page_count)
            super().showPage()
        super().save()

    def _draw_footer(self, page_count):
        width, height = A4
        self.saveState()
        self.setFont("Helvetica", 9)
        self.setFillGray(0.2)
        text = f"Gedruckt am {datetime.now().strftime('%d.%m.%Y %H:%M')}  |  Seite {self._pageNumber} / {page_count}"
        self.drawRightString(width - 12 * mm, 12 * mm, text)
        self.restoreState()


def on_export_pdf(window, _btn=None):
    if not REPORTLAB_AVAILABLE:
        show_message(window, "PDF nicht möglich", "ReportLab ist nicht installiert (pip install reportlab).")
        return

    month = f"{window.kb_month.get_selected() + 1:02d}"
    year_model = window.kb_year.get_model()
    year = year_model.get_string(window.kb_year.get_selected()) if year_model else str(datetime.now().year)

    einnahmen_filter_mode = window.kb_einnahmen_filter.get_selected()

    # Daten sammeln
    ausgaben_rows = []
    for e in window.app.data.get("einkäufe", []):
        d = str(e.get("datum", ""))
        if not d.startswith(f"{year}-{month}"):
            continue
        ausgaben_rows.append([
            _fmt_date_iso_to_de(d),
            e.get("rechnungs_nr") and f"Rechnung {e.get('rechnungs_nr')}" or (e.get("lieferant") or "Einkauf"),
            _fmt_euro(_calc_brutto_from_einkauf(e)),
        ])

    for p in window.app.data.get("privatentnahmen", []):
        d = str(p.get("datum", ""))
        if not d.startswith(f"{year}-{month}"):
            continue
        ausgaben_rows.append([_fmt_date_iso_to_de(d), "Privatentnahme", _fmt_euro(float(p.get("betrag", 0) or 0))])

    ausgaben_rows.sort()

    einnahmen_rows = []
    for v in window.app.data.get("verkäufe", []):
        d = str(v.get("datum", ""))
        if not d.startswith(f"{year}-{month}"):
            continue
            
        hat_rechnung = bool(v.get("rechnung_nr"))
        if einnahmen_filter_mode == 1 and hat_rechnung:
            continue
        if einnahmen_filter_mode == 2 and not hat_rechnung:
            continue

        einnahmen_rows.append([
            _fmt_date_iso_to_de(d),
            v.get("rechnung_nr") and f"Rechnung {v.get('rechnung_nr')}" or "Verkaufsliste",
            _fmt_euro(_calc_brutto_from_verkauf(v)),
        ])

    einnahmen_rows.sort()

    # Summen
    sum_ab = float(window.app.data.get("anfangsbestand", 0.0))

    def _sum_from(rows):
        s = 0.0
        for r in rows:
            s += float(r[2].split()[0].replace(".", "").replace(",", "."))
        return s

    sum_aus = _sum_from(ausgaben_rows)
    sum_ein = _sum_from(einnahmen_rows)
    sum_pri = _sum_from([r for r in ausgaben_rows if r[1] == "Privatentnahme"])

    # Tabellen mit Kopf + Nummerierung
    left_data = [["Nr.:", "Datum", "Beleg", "Betrag"]]
    for i, r in enumerate(ausgaben_rows, start=1):
        left_data.append([str(i), r[0], r[1], r[2]])

    right_data = [["Nr.:", "Datum", "Beleg", "Betrag"]]
    for i, r in enumerate(einnahmen_rows, start=1):
        right_data.append([str(i), r[0], r[1], r[2]])

    # Maximal auffüllen bis 16 Zeilen -> i. d. R. eine Seite
    min_rows = 16
    target_rows = max(min_rows, len(left_data), len(right_data))
    target_rows = min(target_rows, 16)
    while len(left_data) < target_rows:
        left_data.append(["", "", "", ""])
    while len(right_data) < target_rows:
        right_data.append(["", "", "", ""])

    styles = getSampleStyleSheet()
    styles.add(ParagraphStyle(name="H3Center", parent=styles["Heading3"], alignment=1))

    # Kopf zeichnen
    firma = window.app.data.get("stammdaten", {}).get("firma", {})
    firma_name = firma.get("name", "")
    logo_path = Path(__file__).resolve().parent / "icons" / "logo.png"

    def draw_header(canvas, doc):
        width, height = A4
        y = height - 18 * mm

        canvas.setFont("Helvetica-Bold", 14)
        canvas.drawString(12 * mm, y, firma_name)

        if logo_path.exists():
            try:
                canvas.drawImage(
                    str(logo_path), width - 45 * mm, height - 30 * mm,
                    width=30 * mm, height=20 * mm, preserveAspectRatio=True, mask="auto"
                )
            except Exception:
                pass

        canvas.setFont("Helvetica-Bold", 16)
        canvas.drawString(12 * mm, y - 10 * mm, "Kassenbuch")
        canvas.setFont("Helvetica", 10)
        canvas.drawString(12 * mm, y - 16 * mm, f"Monat: {month}/{year}")

        y_pos = y - 24 * mm
        
        # Standard-Schriftart für die meisten Elemente
        canvas.setFont("Helvetica-Bold", 10)
        canvas.drawString(12 * mm, y_pos, "Anfangsbestand:")
        canvas.drawString(64 * mm, y_pos, "Einnahmen:")
        
        canvas.setFont("Helvetica", 10)
        canvas.drawRightString(60 * mm, y_pos, _fmt_euro(sum_ab))
        canvas.drawRightString(106 * mm, y_pos, _fmt_euro(sum_ein))
        
        # --- Speziell für "Ausgaben" & "Privatentnahme" (kleinere Schrift) ---
        canvas.setFont("Helvetica-Bold", 9)
        canvas.drawString(110 * mm, y_pos, "Ausgaben:")
        canvas.drawString(156 * mm, y_pos, "Privatentnahme:")
        
        canvas.setFont("Helvetica", 9)
        canvas.drawRightString(152 * mm, y_pos, _fmt_euro(sum_aus))
        canvas.drawRightString(width - 12 * mm, y_pos, _fmt_euro(sum_pri))
        
        # Trennlinie
        canvas.setLineWidth(0.5)
        canvas.line(12 * mm, y - 27 * mm, width - 12 * mm, y - 27 * mm)

    # Tabellendesign – Spaltenbreiten anpassen
    def build_table(data, total_width):
        tbl = Table(data, colWidths=[10 * mm, 20 * mm, total_width - 55 * mm, 25 * mm])
        tbl.setStyle(TableStyle([
            ("FONT", (0, 0), (-1, 0), "Helvetica-Bold", 9),
            ("ALIGN", (0, 0), (0, -1), "RIGHT"),
            ("ALIGN", (1, 0), (1, -1), "LEFT"),
            ("ALIGN", (2, 0), (2, -1), "LEFT"),
            ("ALIGN", (3, 0), (3, -1), "RIGHT"),
            ("BACKGROUND", (0, 0), (-1, 0), colors.HexColor("#2b6ca3")),
            ("TEXTCOLOR", (0, 0), (-1, 0), colors.white),
            ("GRID", (0, 0), (-1, -1), 0.25, colors.grey),
            ("ROWBACKGROUNDS", (0, 1), (-1, -1), [colors.whitesmoke, colors.Color(1, 1, 1)]),
            ("LEFTPADDING", (0, 0), (-1, -1), 3),
            ("RIGHTPADDING", (0, 0), (-1, -1), 3),
            ("VALIGN", (0, 0), (-1, -1), "MIDDLE"),
            ("TOPPADDING", (0, 0), (-1, -1), 2),
            ("BOTTOMPADDING", (0, 0), (-1, -1), 2),
        ]))
        return tbl

    out_dir = Path(__file__).resolve().parent / "reports"
    out_dir.mkdir(exist_ok=True)
    out_file = out_dir / f"kassenbuch_{year}-{month}.pdf"

    doc = SimpleDocTemplate(
        str(out_file),
        pagesize=A4,
        leftMargin=12 * mm,
        rightMargin=12 * mm,
        topMargin=40 * mm,
        bottomMargin=18 * mm
    )

    col_width = doc.width / 2.0
    left_tbl = build_table(left_data, col_width)
    right_tbl = build_table(right_data, col_width)

    styles.add(ParagraphStyle(name="H3Center2", parent=styles["Heading3"], alignment=1))
    col_titles = Table(
        [[Paragraph("<b>Ausgaben</b>", styles["H3Center"]),
          Paragraph("<b>Einnahmen</b>", styles["H3Center"])]],
        colWidths=[col_width, col_width]
    )
    two_cols = Table([[left_tbl, right_tbl]], colWidths=[col_width, col_width])
    two_cols.setStyle(TableStyle([("VALIGN", (0, 0), (-1, -1), "TOP")]))

    story = [Spacer(1, 5 * mm), col_titles, two_cols]

    doc.build(
        story,
        onFirstPage=draw_header,
        onLaterPages=draw_header,
        canvasmaker=NumberedCanvas
    )

    show_message(window, "PDF erstellt", f"Gespeichert unter:\n{out_file}")