from utils import Gtk, Adw, partial, show_message, REPORTLAB_AVAILABLE, SimpleDocTemplate, Paragraph, Spacer, Table, TableStyle, Image, PageBreak, getSampleStyleSheet, ParagraphStyle, A4, cm, colors, canvas, Path, datetime

def create_kassenbuch_page(window):
    page = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=20)
    page.set_margin_start(40); page.set_margin_end(40)
    page.set_margin_top(40); page.set_margin_bottom(40)

    header_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
    header = Gtk.Label(label="Kassenbuch"); header.add_css_class('title-2'); header.set_halign(Gtk.Align.START); header.set_hexpand(True)
    header_box.append(header)

    pdf_btn = Gtk.Button(label="PDF exportieren")
    pdf_btn.connect('clicked', partial(on_export_pdf, window))
    if not REPORTLAB_AVAILABLE:
        pdf_btn.set_sensitive(False)
        pdf_btn.set_tooltip_text("Python-Bibliothek 'reportlab' nicht gefunden.\nBitte installieren mit: pip install reportlab")
    header_box.append(pdf_btn)

    anfang_btn = Gtk.Button(label="Anfangsbestand setzen"); anfang_btn.connect('clicked', partial(on_set_anfangsbestand, window)); header_box.append(anfang_btn)
    privat_btn = Gtk.Button(label="Privatentnahme"); privat_btn.connect('clicked', partial(on_privatentnahme, window)); header_box.append(privat_btn)
    page.append(header_box)

    monat_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
    monat_label = Gtk.Label(label="Monat:")
    window.kassenbuch_monat = Gtk.DropDown()
    monate = [f"{datetime(2000, i, 1).strftime('%B')} {datetime.now().year}" for i in range(1, 13)]
    window.kassenbuch_monat.set_model(Gtk.StringList.new(monate))
    window.kassenbuch_monat.set_selected(datetime.now().month - 1)
    window.kassenbuch_monat.connect('notify::selected', lambda d, p: refresh_kassenbuch(window))
    monat_box.append(monat_label); monat_box.append(window.kassenbuch_monat); page.append(monat_box)

    scrolled = Gtk.ScrolledWindow(); scrolled.set_vexpand(True); scrolled.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.AUTOMATIC)
    window.kassenbuch_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=20)
    scrolled.set_child(window.kassenbuch_box); page.append(scrolled)

    window.content_stack.add_named(page, 'kassenbuch')

def on_set_anfangsbestand(window, button):
    dialog = Adw.MessageDialog.new(window, "Anfangsbestand setzen",
                                   "Bitte den neuen Anfangsbestand für die Kasse eingeben.")
    dialog.add_response("cancel", "Abbrechen")
    dialog.add_response("save", "Speichern")
    dialog.set_response_appearance("save", Adw.ResponseAppearance.SUGGESTED)

    content = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=10)
    content.set_margin_top(12); content.set_margin_bottom(12)
    entry = Gtk.Entry()
    entry.set_text(str(window.app.data.get('anfangsbestand', 0.0)).replace('.', ','))
    content.append(Gtk.Label(label="Anfangsbestand (€):", xalign=0))
    content.append(entry)
    dialog.set_extra_child(content)

    def on_response(d, response):
        if response == "save":
            try:
                window.app.data['anfangsbestand'] = float(entry.get_text().replace(',', '.'))
                window.app.save_data()
                refresh_kassenbuch(window)
                show_message(window, "Erfolg", "Anfangsbestand gespeichert.")
            except ValueError:
                show_message(window, "Fehler", "Bitte eine gültige Zahl eingeben.")
        d.close()

    dialog.connect("response", on_response)
    dialog.present()

def on_privatentnahme(window, button):
    dialog = Adw.MessageDialog.new(window, "Privatentnahme erfassen", "Geben Sie Datum und Betrag der Entnahme ein.")
    dialog.add_response("cancel", "Abbrechen")
    dialog.add_response("save", "Speichern")
    dialog.set_response_appearance("save", Adw.ResponseAppearance.SUGGESTED)

    content = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=10)
    content.set_margin_top(12); content.set_margin_bottom(12)

    datum_entry = Gtk.Entry()
    datum_entry.set_text(datetime.now().strftime('%Y-%m-%d'))
    content.append(Gtk.Label(label="Datum:", xalign=0))
    content.append(datum_entry)

    betrag_entry = Gtk.Entry()
    betrag_entry.set_text("0,00")
    content.append(Gtk.Label(label="Betrag (€):", xalign=0))
    content.append(betrag_entry)
    
    dialog.set_extra_child(content)

    def on_response(d, response):
        if response == "save":
            try:
                datum = datum_entry.get_text()
                betrag = float(betrag_entry.get_text().replace(',', '.'))
                if betrag <= 0:
                    show_message(window, "Fehler", "Der Betrag muss positiv sein.")
                    return

                window.app.data.setdefault('privatentnahmen', []).append({'datum': datum, 'betrag': betrag})
                window.app.save_data()
                refresh_kassenbuch(window)
                show_message(window, "Erfolg", "Privatentnahme gespeichert.")
            except ValueError:
                show_message(window, "Fehler", "Bitte eine gültige Zahl für den Betrag eingeben.")
        d.close()

    dialog.connect("response", on_response)
    dialog.present()

def refresh_kassenbuch(window):
    if not hasattr(window, 'kassenbuch_box'): return
    while True:
        child = window.kassenbuch_box.get_first_child()
        if child is None: break
        window.kassenbuch_box.remove(child)

    selected = window.kassenbuch_monat.get_selected() if hasattr(window, 'kassenbuch_monat') else datetime.now().month - 1
    monat = (selected or 0) + 1; jahr = datetime.now().year; monat_str = f"{jahr}-{monat:02d}"

    anfangsbestand = window.app.data['anfangsbestand']
    einkäufe_monat = [e for e in window.app.data['einkäufe'] if e.get('datum', '').startswith(monat_str)]
    ausgaben = sum(e.get('gesamt_brutto', e.get('gesamt', 0)) for e in einkäufe_monat)
    verkäufe_monat = [v for v in window.app.data['verkäufe'] if v.get('datum', '').startswith(monat_str)]
    einnahmen = sum(v.get('gesamt', 0) for v in verkäufe_monat)
    privatentnahmen_monat = [p for p in window.app.data['privatentnahmen'] if p.get('datum', '').startswith(monat_str)]
    privatentnahmen = sum(p.get('betrag', 0) for p in privatentnahmen_monat)
    kassenbestand = anfangsbestand + einnahmen - ausgaben - privatentnahmen

    summary_grid = Gtk.Grid(); summary_grid.set_row_spacing(15); summary_grid.set_column_spacing(15); summary_grid.set_column_homogeneous(True)
    summary_grid.attach(create_summary_card("Anfangsbestand", f"{anfangsbestand:.2f} €", "blue"), 0, 0, 1, 1)
    summary_grid.attach(create_summary_card("Einnahmen", f"{einnahmen:.2f} €", "green"), 1, 0, 1, 1)
    summary_grid.attach(create_summary_card("Ausgaben", f"{ausgaben:.2f} €", "red"), 0, 1, 1, 1)
    summary_grid.attach(create_summary_card("Privatentnahmen", f"{privatentnahmen:.2f} €", "orange"), 1, 1, 1, 1)
    window.kassenbuch_box.append(summary_grid)

    bestand_frame = Gtk.Frame(); bestand_frame.set_margin_top(10)
    bestand_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5)
    bestand_box.set_margin_start(15); bestand_box.set_margin_end(15); bestand_box.set_margin_top(15); bestand_box.set_margin_bottom(15)
    bestand_label = Gtk.Label(label="Aktueller Kassenbestand"); bestand_label.add_css_class('title-3'); bestand_box.append(bestand_label)
    bestand_value = Gtk.Label(label=f"{kassenbestand:.2f} €"); bestand_value.add_css_class('title-1')
    if kassenbestand < 0: bestand_value.add_css_class('error')
    else: bestand_value.add_css_class('success')
    bestand_box.append(bestand_value); bestand_frame.set_child(bestand_box); window.kassenbuch_box.append(bestand_frame)

    if einkäufe_monat:
        einkauf_label = Gtk.Label(label="Einkäufe des Monats"); einkauf_label.add_css_class('title-3'); einkauf_label.set_halign(Gtk.Align.START); einkauf_label.set_margin_top(20)
        window.kassenbuch_box.append(einkauf_label)
        einkauf_list = Gtk.ListBox(); einkauf_list.add_css_class('boxed-list')
        for einkauf in einkäufe_monat:
            row = Gtk.ListBoxRow()
            box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
            box.set_margin_start(10); box.set_margin_end(10); box.set_margin_top(8); box.set_margin_bottom(8)
            datum = Gtk.Label(label=einkauf.get('datum', '')); datum.set_size_request(100, -1); datum.set_halign(Gtk.Align.START); box.append(datum)
            
            artikel_text = einkauf.get('artikel', '')
            if re_nr := einkauf.get('rechnungs_nr'):
                artikel_text += f" (Re-Nr: {re_nr})"
            artikel = Gtk.Label(label=artikel_text)
            artikel.set_halign(Gtk.Align.START); artikel.set_hexpand(True); box.append(artikel)
            
            betrag_val = einkauf.get('gesamt_brutto', einkauf.get('gesamt', 0))
            betrag = Gtk.Label(label=f"-{betrag_val:.2f} €"); betrag.add_css_class('error'); box.append(betrag)
            row.set_child(box); einkauf_list.append(row)
        window.kassenbuch_box.append(einkauf_list)

    if verkäufe_monat:
        verkauf_label = Gtk.Label(label="Verkäufe des Monats"); verkauf_label.add_css_class('title-3'); verkauf_label.set_halign(Gtk.Align.START); verkauf_label.set_margin_top(20)
        window.kassenbuch_box.append(verkauf_label)
        verkauf_list = Gtk.ListBox(); verkauf_list.add_css_class('boxed-list')
        for verkauf in verkäufe_monat:
            row = Gtk.ListBoxRow()
            box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
            box.set_margin_start(10); box.set_margin_end(10); box.set_margin_top(8); box.set_margin_bottom(8)
            datum = Gtk.Label(label=verkauf.get('datum', '')); datum.set_size_request(100, -1); datum.set_halign(Gtk.Align.START); box.append(datum)
            
            artikel_text = verkauf.get('artikel', '')
            if kunde := verkauf.get('kunde'):
                artikel_text += f" ({kunde})"
            artikel = Gtk.Label(label=artikel_text)
            artikel.set_halign(Gtk.Align.START); artikel.set_hexpand(True); box.append(artikel)
            
            betrag = Gtk.Label(label=f"+{verkauf.get('gesamt', 0):.2f} €"); betrag.add_css_class('success'); box.append(betrag)
            row.set_child(box); verkauf_list.append(row)
        window.kassenbuch_box.append(verkauf_list)

    if privatentnahmen_monat:
        privat_label = Gtk.Label(label="Privatentnahmen des Monats"); privat_label.add_css_class('title-3'); privat_label.set_halign(Gtk.Align.START); privat_label.set_margin_top(20)
        window.kassenbuch_box.append(privat_label)
        privat_list = Gtk.ListBox(); privat_list.add_css_class('boxed-list')
        for entnahme in privatentnahmen_monat:
            row = Gtk.ListBoxRow()
            box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
            box.set_margin_start(10); box.set_margin_end(10); box.set_margin_top(8); box.set_margin_bottom(8)
            datum = Gtk.Label(label=entnahme.get('datum', '')); datum.set_size_request(100, -1); datum.set_halign(Gtk.Align.START); box.append(datum)
            text = Gtk.Label(label="Privatentnahme"); text.set_halign(Gtk.Align.START); text.set_hexpand(True); box.append(text)
            betrag = Gtk.Label(label=f"-{entnahme.get('betrag', 0):.2f} €"); betrag.add_css_class('warning'); box.append(betrag)
            row.set_child(box); privat_list.append(row)
        window.kassenbuch_box.append(privat_list)

def on_export_pdf(window, button):
    dialog = Gtk.FileChooserDialog(
        title="PDF speichern unter...",
        transient_for=window,
        action=Gtk.FileChooserAction.SAVE,
    )
    dialog.add_buttons(
        "Abbrechen", Gtk.ResponseType.CANCEL,
        "Speichern", Gtk.ResponseType.ACCEPT
    )
    
    selected_month_str = window.kassenbuch_monat.get_model().get_string(window.kassenbuch_monat.get_selected())
    default_filename = f"Kassenbuch_{selected_month_str.replace(' ', '_')}.pdf"
    dialog.set_current_name(default_filename)

    def on_response(d, response_id):
        if response_id == Gtk.ResponseType.ACCEPT:
            filepath = d.get_file().get_path()
            if not filepath.lower().endswith('.pdf'):
                filepath += '.pdf'
            
            try:
                generate_kassenbuch_pdf(window, filepath)
                show_message(window, "Erfolg", f"Das Kassenbuch wurde erfolgreich als PDF gespeichert:\n{filepath}")
            except Exception as e:
                show_message(window, "Fehler beim PDF-Export", f"Ein Fehler ist aufgetreten:\n{e}")
        
        d.destroy()

    dialog.connect("response", on_response)
    dialog.present()

def generate_kassenbuch_pdf(window, filepath):
    if not REPORTLAB_AVAILABLE:
        raise RuntimeError("Die ReportLab-Bibliothek ist für den PDF-Export erforderlich.")

    doc = SimpleDocTemplate(filepath, pagesize=A4,
                            topMargin=3.0*cm, bottomMargin=2.5*cm,
                            leftMargin=2*cm, rightMargin=2*cm)
    
    story = []
    styles = getSampleStyleSheet()
    styles.add(ParagraphStyle(name='RightAlign', alignment=2))
    
    # Daten für PDF sammeln
    firma = window.app.data['stammdaten']['firma']
    selected_idx = window.kassenbuch_monat.get_selected()
    monat_nr = selected_idx + 1
    jahr = datetime.now().year
    monat_name = datetime(2000, monat_nr, 1).strftime('%B')
    monat_str = f"{jahr}-{monat_nr:02d}"

    anfangsbestand = window.app.data.get('anfangsbestand', 0.0)
    einkäufe_monat = [e for e in window.app.data['einkäufe'] if e.get('datum', '').startswith(monat_str)]
    
    ausgaben = sum(e.get('gesamt_brutto', e.get('gesamt', 0)) for e in einkäufe_monat)
    
    verkäufe_monat = [v for v in window.app.data['verkäufe'] if v.get('datum', '').startswith(monat_str)]
    einnahmen = sum(v.get('gesamt', 0) for v in verkäufe_monat)
    privatentnahmen_monat = [p for p in window.app.data['privatentnahmen'] if p.get('datum', '').startswith(monat_str)]
    privatentnahmen = sum(p.get('betrag', 0) for p in privatentnahmen_monat)
    kassenbestand = anfangsbestand + einnahmen - ausgaben - privatentnahmen

    # --- PDF-Inhalt (Story) erstellen ---
    story.append(Paragraph(f"Kassenbuch für {monat_name} {jahr}", styles['h1']))
    story.append(Spacer(1, 1*cm))
    
    right_aligned_style = styles['RightAlign']
    summary_data = [
        [Paragraph('<b>Anfangsbestand</b>', styles['Normal']), Paragraph(f"{anfangsbestand:,.2f} €", right_aligned_style)],
        [Paragraph('<b>Einnahmen</b>', styles['Normal']), Paragraph(f"<font color='green'>{einnahmen:,.2f} €</font>", right_aligned_style)],
        [Paragraph('<b>Ausgaben</b>', styles['Normal']), Paragraph(f"<font color='red'>-{ausgaben:,.2f} €</font>", right_aligned_style)],
        [Paragraph('<b>Privatentnahmen</b>', styles['Normal']), Paragraph(f"<font color='orange'>-{privatentnahmen:,.2f} €</font>", right_aligned_style)],
        [Paragraph('<b>Aktueller Kassenbestand</b>', styles['h3']), Paragraph(f"<b>{kassenbestand:,.2f} €</b>", styles['h3'])]
    ]
    
    summary_table = Table(summary_data, colWidths=[doc.width/2.0]*2)
    summary_table.setStyle(TableStyle([('VALIGN', (0, 0), (-1, -1), 'MIDDLE'), ('LINEBELOW', (0, -2), (1, -2), 1, colors.grey), ('TOPPADDING', (0,0), (-1,-1), 8), ('BOTTOMPADDING', (0,0), (-1,-1), 8)]))
    story.append(summary_table)
    story.append(Spacer(1, 1.5*cm))

    table_style = TableStyle([('BACKGROUND', (0, 0), (-1, 0), colors.lightgrey), ('TEXTCOLOR', (0, 0), (-1, 0), colors.black), ('ALIGN', (0, 0), (-1, -1), 'CENTER'), ('FONTNAME', (0, 0), (-1, 0), 'Helvetica-Bold'), ('BOTTOMPADDING', (0, 0), (-1, 0), 12), ('BACKGROUND', (0, 1), (-1, -1), colors.whitesmoke), ('GRID', (0, 0), (-1, -1), 1, colors.black), ('ALIGN', (1, 1), (1, -1), 'LEFT')])
    
    def create_transaction_table(title, data, headers, colWidths, row_formatter):
        if data:
            story.append(Paragraph(title, styles['h2']))
            story.append(Spacer(1, 0.5*cm))
            table_data = [headers] + [row_formatter(item) for item in data]
            t = Table(table_data, colWidths=colWidths)
            t.setStyle(table_style)
            story.append(t)
            story.append(Spacer(1, 1*cm))

    create_transaction_table("Einkäufe des Monats", einkäufe_monat, ['Datum', 'Artikel / Rechnungs-Nr.', 'Lieferant', 'Betrag'], [2.5*cm, 8.5*cm, 4*cm, 3*cm], lambda e: [e.get('datum', ''), Paragraph(f"{e.get('artikel', '')}<br/><font size='8' color='grey'>Re-Nr: {e.get('rechnungs_nr', 'k.A.')}</font>"), e.get('lieferant', ''), f"-{e.get('gesamt_brutto', e.get('gesamt', 0)):.2f} €"])
    create_transaction_table("Verkäufe des Monats", verkäufe_monat, ['Datum', 'Artikel', 'Kunde', 'Betrag'], [2.5*cm, 8.5*cm, 4*cm, 3*cm], lambda v: [v.get('datum', ''), v.get('artikel', ''), v.get('kunde', ''), f"+{v.get('gesamt', 0):.2f} €"])
    create_transaction_table("Privatentnahmen des Monats", privatentnahmen_monat, ['Datum', 'Beschreibung', 'Betrag'], [2.5*cm, 12.5*cm, 3*cm], lambda p: [p.get('datum', ''), 'Privatentnahme', f"-{p.get('betrag', 0):.2f} €"])

    def draw_header_footer(canvas, doc):
        canvas.saveState()
        
        y_pos = doc.pagesize[1] - 1.5 * cm 

        logo_path = Path(__file__).resolve().parent / 'icons' / 'logo.png'
        if logo_path.exists():
            try:
                img = Image(logo_path, width=1.8*cm, height=1.8*cm)
                img.drawOn(canvas, doc.leftMargin, y_pos - 0.4*cm) 
                
                canvas.setFont('Helvetica-Bold', 14)
                canvas.drawString(doc.leftMargin + 2.2*cm, y_pos, firma.get('name', ''))
            except Exception:
                canvas.setFont('Helvetica-Bold', 16)
                canvas.drawString(doc.leftMargin, y_pos, firma.get('name', ''))
        else:
            canvas.setFont('Helvetica-Bold', 16)
            canvas.drawString(doc.leftMargin, y_pos, firma.get('name', ''))

        canvas.setFont('Helvetica', 9)
        address = f"{firma.get('strasse', '')}, {firma.get('plz', '')} {firma.get('ort', '')}"
        canvas.drawRightString(doc.width + doc.leftMargin, y_pos, address)
        
        footer_text = f"Gedruckt am: {datetime.now().strftime('%d.%m.%Y')}"
        canvas.drawString(doc.leftMargin, 1.5*cm, footer_text)
        page_num_text = f"Seite {doc.page}" 
        canvas.drawRightString(doc.width + doc.leftMargin, 1.5*cm, page_num_text)
        
        canvas.restoreState()
    
    doc.build(story, onFirstPage=draw_header_footer, onLaterPages=draw_header_footer)

def create_summary_card(title, value, color):
    frame = Gtk.Frame()
    box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5)
    box.set_margin_start(15); box.set_margin_end(15); box.set_margin_top(15); box.set_margin_bottom(15)
    title_label = Gtk.Label(label=title); title_label.add_css_class('dim-label'); title_label.set_halign(Gtk.Align.START); box.append(title_label)
    value_label = Gtk.Label(label=value); value_label.add_css_class('title-2'); value_label.set_halign(Gtk.Align.START)
    if color == "green": value_label.add_css_class('success')
    elif color == "red": value_label.add_css_class('error')
    elif color == "orange": value_label.add_css_class('warning')
    box.append(value_label); frame.set_child(box); return frame