# rechnung.py (updated with ZUGFeRD support and payment terms)
from utils import Gtk, Adw, partial, show_message, REPORTLAB_AVAILABLE, SimpleDocTemplate, Paragraph, Spacer, Table, TableStyle, getSampleStyleSheet, ParagraphStyle, A4, cm, colors, Path, datetime
import lxml.etree as ET

try:
    from facturx import generate_from_file
    FACTURX_AVAILABLE = True
except ImportError:
    FACTURX_AVAILABLE = False

def create_rechnung_page(window):
    page = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=20)
    page.set_margin_start(40); page.set_margin_end(40)
    page.set_margin_top(40); page.set_margin_bottom(40)
    header = Gtk.Label(label="Rechnung erstellen"); header.add_css_class('title-2'); header.set_halign(Gtk.Align.START); page.append(header)

    filter_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
    filter_label = Gtk.Label(label="Filter:")
    window.rechnung_filter = Gtk.Entry(); window.rechnung_filter.set_placeholder_text("Artikel oder Kunde suchen...")
    window.rechnung_filter.connect('changed', lambda e: refresh_rechnung_list(window))
    filter_box.append(filter_label); filter_box.append(window.rechnung_filter); page.append(filter_box)

    scrolled = Gtk.ScrolledWindow(); scrolled.set_vexpand(True); scrolled.set_policy(Gtk.PolicyType.NEVER, Gtk.PolicyType.AUTOMATIC)
    window.rechnung_list_box = Gtk.ListBox(); window.rechnung_list_box.add_css_class('boxed-list')
    scrolled.set_child(window.rechnung_list_box); page.append(scrolled)

    options_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
    window.zugferd_check = Gtk.CheckButton(label="Als ZUGFeRD E-Rechnung (PDF + XML)")
    if not FACTURX_AVAILABLE or not REPORTLAB_AVAILABLE:
        window.zugferd_check.set_sensitive(False)
        window.zugferd_check.set_tooltip_text("factur-x Bibliothek oder ReportLab fehlt. Installiere mit 'pip install factur-x'.")
    options_box.append(window.zugferd_check)
    page.append(options_box)

    ## NEU: Eingabefeld für das Zahlungsziel hinzufügen ##
    zahlungsziel_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10, margin_bottom=15, margin_top=10)
    zahlungsziel_label = Gtk.Label(label="Zahlungsziel:")
    window.rechnung_zahlungsziel_entry = Gtk.Entry()
    window.rechnung_zahlungsziel_entry.set_placeholder_text("z.B. Zahlbar sofort ohne Abzug")
    window.rechnung_zahlungsziel_entry.set_hexpand(True)
    # Standardtext setzen
    window.rechnung_zahlungsziel_entry.set_text("Zahlbar innerhalb von 14 Tagen ohne Abzug.")
    
    zahlungsziel_box.append(zahlungsziel_label)
    zahlungsziel_box.append(window.rechnung_zahlungsziel_entry)
    page.append(zahlungsziel_box)

    btn = Gtk.Button(label="Ausgewählte Rechnung erstellen"); btn.add_css_class('suggested-action')
    btn.connect('clicked', partial(on_create_rechnung, window)); page.append(btn)

    window.content_stack.add_named(page, 'rechnung')
    refresh_rechnung_list(window)

def refresh_rechnung_list(window):
    while window.rechnung_list_box.get_row_at_index(0):
        window.rechnung_list_box.remove(window.rechnung_list_box.get_row_at_index(0))

    filter_text = window.rechnung_filter.get_text().lower()
    for verkauf in reversed(window.app.data['verkäufe']):
        if filter_text and not (filter_text in verkauf['artikel'].lower() or filter_text in verkauf.get('kunde', '').lower()):
            continue
        row = Gtk.ListBoxRow()
        box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=5)
        box.set_margin_start(10); box.set_margin_end(10); box.set_margin_top(10); box.set_margin_bottom(10)
        header_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=10)
        check = Gtk.CheckButton(); check.set_valign(Gtk.Align.CENTER)
        header_box.append(check)
        title = Gtk.Label(label=f"{verkauf['artikel']} - {verkauf['menge']} {verkauf['einheit']}"); title.add_css_class('heading')
        title.set_halign(Gtk.Align.START); title.set_hexpand(True); header_box.append(title)
        preis_label = Gtk.Label(label=f"{verkauf['gesamt']:.2f} €"); preis_label.add_css_class('title-3'); header_box.append(preis_label)
        box.append(header_box)
        details = Gtk.Label(label=f"{verkauf['datum']} | {verkauf['preis']:.4f} €/Einheit | MwSt: {verkauf['steuer']}% | {verkauf.get('kunde', '')}")
        details.add_css_class('dim-label'); details.set_halign(Gtk.Align.START); box.append(details)
        row.set_child(box)
        row.verkauf = verkauf
        row.check = check
        window.rechnung_list_box.append(row)

def on_create_rechnung(window, button):
    selected = []
    i = 0
    row = window.rechnung_list_box.get_row_at_index(i)
    while row:
        if row.check.get_active():
            selected.append(row.verkauf)
        i += 1
        row = window.rechnung_list_box.get_row_at_index(i)
    
    if not selected:
        show_message(window, "Fehler", "Keine Verkäufe ausgewählt.")
        return
    
    kunden = set(v.get('kunde', '') for v in selected if v.get('kunde'))
    if len(kunden) > 1:
        show_message(window, "Fehler", "Ausgewählte Verkäufe haben unterschiedliche Kunden. Bitte nur Verkäufe eines Kunden auswählen.")
        return
    kunde_name = next(iter(kunden)) if kunden else ''
    if not kunde_name:
        show_message(window, "Fehler", "Kein Kunde für die ausgewählten Verkäufe angegeben.")
        return
    
    kunde = next((k for k in window.app.data['stammdaten']['kunden'] if k['name'] == kunde_name), None)
    if not kunde:
        show_message(window, "Fehler", "Kundendaten nicht gefunden.")
        return

    ## NEU: Text aus dem Zahlungsziel-Feld auslesen ##
    zahlungsziel_text = window.rechnung_zahlungsziel_entry.get_text()

    dialog = Gtk.FileChooserDialog(
        title="Rechnung speichern unter...",
        transient_for=window,
        action=Gtk.FileChooserAction.SAVE,
    )
    dialog.add_buttons("Abbrechen", Gtk.ResponseType.CANCEL, "Speichern", Gtk.ResponseType.ACCEPT)
    default_filename = f"Rechnung_{kunde_name.replace(' ', '_')}_{datetime.now().strftime('%Y%m%d')}.pdf"
    dialog.set_current_name(default_filename)

    def on_response(d, response_id):
        if response_id == Gtk.ResponseType.ACCEPT:
            filepath = d.get_file().get_path()
            if not filepath.lower().endswith('.pdf'):
                filepath += '.pdf'
            try:
                ## GEÄNDERT: zahlungsziel_text wird übergeben ##
                pdf_path = generate_rechnung_pdf(window, selected, kunde, filepath, zahlungsziel_text)
                if window.zugferd_check.get_active() and FACTURX_AVAILABLE:
                    xml_path = generate_zugferd_xml(window, selected, kunde, filepath)
                    zugferd_path = embed_zugferd_xml(pdf_path, xml_path)
                    show_message(window, "Erfolg", f"ZUGFeRD Rechnung gespeichert: {zugferd_path}\nXML-Datei: {xml_path}")
                else:
                    show_message(window, "Erfolg", f"Rechnung gespeichert: {pdf_path}")
            except Exception as e:
                show_message(window, "Fehler", f"Fehler beim Erstellen der Rechnung: {e}")
        d.destroy()

    dialog.connect("response", on_response)
    dialog.present()

## GEÄNDERT: Funktion akzeptiert jetzt zahlungsziel_text ##
def generate_rechnung_pdf(window, selected, kunde, filepath, zahlungsziel_text):
    if not REPORTLAB_AVAILABLE:
        raise RuntimeError("ReportLab nicht verfügbar.")

    doc = SimpleDocTemplate(filepath, pagesize=A4, topMargin=3*cm, bottomMargin=2.5*cm, leftMargin=2*cm, rightMargin=2*cm)
    story = []
    styles = getSampleStyleSheet()
    styles.add(ParagraphStyle(name='RightAlign', alignment=2))

    firma = window.app.data['stammdaten']['firma']

    story.append(Paragraph("Rechnung", styles['h1']))
    story.append(Spacer(1, 0.5*cm))
    story.append(Paragraph(f"Rechnungsdatum: {datetime.now().strftime('%d.%m.%Y')}", styles['Normal']))
    story.append(Spacer(1, 1*cm))

    story.append(Paragraph("<b>Absender:</b>", styles['Normal']))
    story.append(Paragraph(firma.get('name', ''), styles['Normal']))
    story.append(Paragraph(firma.get('strasse', ''), styles['Normal']))
    story.append(Paragraph(f"{firma.get('plz', '')} {firma.get('ort', '')}", styles['Normal']))
    story.append(Paragraph(f"Email: {firma.get('email', '')}", styles['Normal']))
    story.append(Paragraph(f"Telefon: {firma.get('telefon', '')}", styles['Normal']))
    story.append(Spacer(1, 1*cm))

    story.append(Paragraph("<b>Empfänger:</b>", styles['Normal']))
    story.append(Paragraph(kunde.get('name', ''), styles['Normal']))
    story.append(Paragraph(kunde.get('strasse', ''), styles['Normal']))
    story.append(Paragraph(f"{kunde.get('plz', '')} {kunde.get('ort', '')}", styles['Normal']))
    story.append(Paragraph(f"Email: {kunde.get('email', '')}", styles['Normal']))
    story.append(Paragraph(f"Telefon: {kunde.get('telefon', '')}", styles['Normal']))
    story.append(Spacer(1, 1*cm))

    data = [['Artikel', 'Menge', 'Einheit', 'Preis pro Einheit (€)', 'Steuer (%)', 'Gesamt (€)']]
    gesamt_netto = 0
    steuer_betraege = {}
    for v in selected:
        netto_pro_position = v['menge'] * v['preis']
        gesamt_brutto = v['gesamt']
        steuer_satz = v['steuer']
        steuer_betrag = gesamt_brutto - netto_pro_position
        gesamt_netto += netto_pro_position
        if steuer_satz not in steuer_betraege: steuer_betraege[steuer_satz] = 0
        steuer_betraege[steuer_satz] += steuer_betrag
        artikel_paragraph = Paragraph(v['artikel'], styles['Normal'])
        data.append([artikel_paragraph, str(v['menge']), v['einheit'], f"{v['preis']:.2f}", str(v['steuer']), f"{gesamt_brutto:.2f}"])

    col_widths = [5*cm, 1.8*cm, 1.8*cm, 4.8*cm, 2*cm, 2.2*cm]
    table = Table(data, colWidths=col_widths)

    table.setStyle(TableStyle([
        ('BACKGROUND', (0,0), (-1,0), colors.darkgray),
        ('TEXTCOLOR',(0,0),(-1,0), colors.whitesmoke),
        ('FONTNAME', (0,0), (-1,0), 'Helvetica-Bold'),
        ('BOTTOMPADDING', (0,0), (-1,0), 12),
        ('BACKGROUND',(0,1),(-1,-1), colors.white),
        ('ALIGN',(1,0),(-1,-1),'CENTER'),
        ('ALIGN', (0, 1), (0, -1), 'LEFT'),
        ('VALIGN',(0,0),(-1,-1),'MIDDLE'),
        ('GRID',(0,0),(-1,-1),0.25,colors.white)
    ]))
    story.append(table)
    story.append(Spacer(1, 1*cm))

    story.append(Paragraph(f"Gesamt Netto: {gesamt_netto:.2f} €", styles['Normal']))
    for steuer, betrag in steuer_betraege.items():
        if betrag > 0:
            story.append(Paragraph(f"MwSt. {steuer}%: {betrag:.2f} €", styles['Normal']))
    story.append(Paragraph(f"<b>Gesamtbetrag Brutto: {sum(v['gesamt'] for v in selected):.2f} €</b>", styles['Normal']))

    ## NEU: Zahlungsziel zur PDF hinzufügen, falls vorhanden ##
    if zahlungsziel_text:
        story.append(Spacer(1, 1*cm))
        story.append(Paragraph(zahlungsziel_text, styles['Normal']))
    ## NEU: Bankverbindung hinzufügen ##
        story.append(Spacer(1, 1*cm))
        bank_details_parts = []
    if firma.get('bank'):
        bank_details_parts.append(f"<b>Bank:</b> {firma.get('bank')}")
    if firma.get('iban'):
        bank_details_parts.append(f"<b>IBAN:</b> {firma.get('iban')}")
    if firma.get('bic'):
        bank_details_parts.append(f"<b>BIC:</b> {firma.get('bic')}")

    if bank_details_parts:
        bank_details_text = " | ".join(bank_details_parts)
        story.append(Paragraph("<b>Unsere Bankverbindung:</b>", styles['Normal']))
        story.append(Paragraph(bank_details_text, styles['Normal']))    

    def draw_header_footer(canvas, doc):
        canvas.saveState()
        
        y_pos = doc.pagesize[1] - 1.5 * cm 
        logo_path = Path(__file__).resolve().parent / 'icons' / 'logo.png'
        if logo_path.exists():
            try:
                img_width = 1.8 * cm
                img_height = 1.8 * cm
                canvas.drawInlineImage(str(logo_path), doc.width + doc.leftMargin - img_width, y_pos - 0.4 * cm, width=img_width, height=img_height)
            except Exception:
                pass

        canvas.setFont('Helvetica', 9)
        canvas.drawString(cm, cm, f"Seite {doc.page}")
        canvas.restoreState()

    doc.build(story, onFirstPage=draw_header_footer, onLaterPages=draw_header_footer)
    return filepath

def generate_zugferd_xml(window, selected, kunde, filepath):
    # ... (diese Funktion bleibt unverändert)
    NS = {
        'rsm': "urn:un:unece:uncefact:data:standard:CrossIndustryInvoice:100",
        'ram': "urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100",
        'udt': "urn:un:unece:uncefact:data:standard:UnqualifiedDataType:100"
    }

    root = ET.Element("{urn:un:unece:uncefact:data:standard:CrossIndustryInvoice:100}CrossIndustryInvoice", nsmap=NS)
    
    context = ET.SubElement(root, "{urn:un:unece:uncefact:data:standard:CrossIndustryInvoice:100}ExchangedDocumentContext")
    param = ET.SubElement(context, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}GuidelineSpecifiedDocumentContextParameter")
    param_id = ET.SubElement(param, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}ID")
    param_id.text = "urn:cen.eu:en16931:2017"

    header = ET.SubElement(root, "{urn:un:unece:uncefact:data:standard:CrossIndustryInvoice:100}ExchangedDocument")
    inv_id = ET.SubElement(header, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}ID")
    inv_id.text = f"INV-{datetime.now().strftime('%Y%m%d%H%M')}"
    type_code = ET.SubElement(header, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}TypeCode")
    type_code.text = "380"
    issue_dt = ET.SubElement(header, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}IssueDateTime")
    dt_str = ET.SubElement(issue_dt, "{urn:un:unece:uncefact:data:standard:UnqualifiedDataType:100}DateTimeString", format="102")
    dt_str.text = datetime.now().strftime('%Y%m%d')

    transaction = ET.SubElement(root, "{urn:un:unece:uncefact:data:standard:CrossIndustryInvoice:100}SupplyChainTradeTransaction")
    
    for idx, v in enumerate(selected, start=1):
        line_item = ET.SubElement(transaction, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}IncludedSupplyChainTradeLineItem")
        doc_line = ET.SubElement(line_item, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}AssociatedDocumentLineDocument")
        line_id = ET.SubElement(doc_line, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}LineID")
        line_id.text = str(idx)
        product = ET.SubElement(line_item, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}SpecifiedTradeProduct")
        prod_name = ET.SubElement(product, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}Name")
        prod_name.text = v['artikel']
        trade_agreement = ET.SubElement(line_item, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}SpecifiedLineTradeAgreement")
        net_price = ET.SubElement(trade_agreement, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}NetPriceProductTradePrice")
        charge_amt = ET.SubElement(net_price, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}ChargeAmount", currencyID="EUR")
        charge_amt.text = f"{v['preis']:.4f}"
        trade_delivery = ET.SubElement(line_item, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}SpecifiedLineTradeDelivery")
        billed_qty = ET.SubElement(trade_delivery, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}BilledQuantity", unitCode=v['einheit'].upper())
        billed_qty.text = f"{v['menge']:.4f}"
        trade_settlement = ET.SubElement(line_item, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}SpecifiedLineTradeSettlement")
        appl_tax = ET.SubElement(trade_settlement, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}ApplicableTradeTax")
        type_code = ET.SubElement(appl_tax, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}TypeCode")
        type_code.text = "VAT"
        category = ET.SubElement(appl_tax, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}CategoryCode")
        category.text = "S"
        percent = ET.SubElement(appl_tax, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}RateApplicablePercent")
        percent.text = f"{v['steuer']:.2f}"
        line_summ = ET.SubElement(trade_settlement, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}SpecifiedTradeSettlementLineMonetarySummation")
        line_total_amount = v['menge'] * v['preis']
        line_total = ET.SubElement(line_summ, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}LineTotalAmount", currencyID="EUR")
        line_total.text = f"{line_total_amount:.2f}"
        
    agreement = ET.SubElement(transaction, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}ApplicableHeaderTradeAgreement")
    seller = ET.SubElement(agreement, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}SellerTradeParty")
    seller_name = ET.SubElement(seller, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}Name")
    seller_name.text = window.app.data['stammdaten']['firma'].get('name', '')
    seller_addr = ET.SubElement(seller, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}PostalTradeAddress")
    postcode = ET.SubElement(seller_addr, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}PostcodeCode")
    postcode.text = window.app.data['stammdaten']['firma'].get('plz', '')
    line_one = ET.SubElement(seller_addr, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}LineOne")
    line_one.text = window.app.data['stammdaten']['firma'].get('strasse', '')
    city = ET.SubElement(seller_addr, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}CityName")
    city.text = window.app.data['stammdaten']['firma'].get('ort', '')
    country = ET.SubElement(seller_addr, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}CountryID")
    country.text = "DE"
    buyer = ET.SubElement(agreement, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}BuyerTradeParty")
    buyer_name = ET.SubElement(buyer, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}Name")
    buyer_name.text = kunde.get('name', '')
    buyer_addr = ET.SubElement(buyer, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}PostalTradeAddress")
    postcode = ET.SubElement(buyer_addr, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}PostcodeCode")
    postcode.text = kunde.get('plz', '')
    line_one = ET.SubElement(buyer_addr, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}LineOne")
    line_one.text = kunde.get('strasse', '')
    city = ET.SubElement(buyer_addr, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}CityName")
    city.text = kunde.get('ort', '')
    country = ET.SubElement(buyer_addr, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}CountryID")
    country.text = "DE"

    delivery = ET.SubElement(transaction, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}ApplicableHeaderTradeDelivery")
    
    settlement = ET.SubElement(transaction, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}ApplicableHeaderTradeSettlement")
    currency = ET.SubElement(settlement, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}InvoiceCurrencyCode")
    currency.text = "EUR"

    steuer_betraege = {}
    gesamt_netto = 0
    for v in selected:
        netto_pro_position = v['menge'] * v['preis']
        steuer_satz = v['steuer']
        steuer_betrag = v['gesamt'] - netto_pro_position
        gesamt_netto += netto_pro_position
        if steuer_satz not in steuer_betraege: steuer_betraege[steuer_satz] = {'basis': 0, 'betrag': 0}
        steuer_betraege[steuer_satz]['basis'] += netto_pro_position
        steuer_betraege[steuer_satz]['betrag'] += steuer_betrag

    for steuer, werte in steuer_betraege.items():
        tax = ET.SubElement(settlement, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}ApplicableTradeTax")
        calc_amt = ET.SubElement(tax, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}CalculatedAmount", currencyID="EUR")
        calc_amt.text = f"{werte['betrag']:.2f}"
        type_code = ET.SubElement(tax, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}TypeCode")
        type_code.text = "VAT"
        basis_amt = ET.SubElement(tax, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}BasisAmount", currencyID="EUR")
        basis_amt.text = f"{werte['basis']:.2f}"
        category = ET.SubElement(tax, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}CategoryCode")
        category.text = "S"
        percent = ET.SubElement(tax, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}RateApplicablePercent")
        percent.text = f"{steuer:.2f}"

    gesamt_steuer = sum(w['betrag'] for w in steuer_betraege.values())
    gesamt_brutto = gesamt_netto + gesamt_steuer
    summation = ET.SubElement(settlement, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}SpecifiedTradeSettlementHeaderMonetarySummation")
    line_total = ET.SubElement(summation, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}LineTotalAmount", currencyID="EUR")
    line_total.text = f"{gesamt_netto:.2f}"
    tax_basis_total = ET.SubElement(summation, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}TaxBasisTotalAmount", currencyID="EUR")
    tax_basis_total.text = f"{gesamt_netto:.2f}"
    tax_total = ET.SubElement(summation, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}TaxTotalAmount", currencyID="EUR")
    tax_total.text = f"{gesamt_steuer:.2f}"
    grand_total = ET.SubElement(summation, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}GrandTotalAmount", currencyID="EUR")
    grand_total.text = f"{gesamt_brutto:.2f}"
    due_payable = ET.SubElement(summation, "{urn:un:unece:uncefact:data:standard:ReusableAggregateBusinessInformationEntity:100}DuePayableAmount", currencyID="EUR")
    due_payable.text = f"{gesamt_brutto:.2f}"

    xml_path = filepath.replace('.pdf', '.xml')
    tree = ET.ElementTree(root)
    tree.write(xml_path, encoding='utf-8', xml_declaration=True, pretty_print=True)
    return xml_path

def embed_zugferd_xml(pdf_path, xml_path):
    zugferd_pdf_path = pdf_path.replace('.pdf', '_zugferd.pdf')
    generate_from_file(pdf_path, xml_path, output_pdf_file=zugferd_pdf_path, level='basic')
    return zugferd_pdf_path