import sqlite3
from pathlib import Path
from datetime import datetime

DEFAULT_FIRMA = {
    "name": "", "strasse": "", "plz": "", "ort": "",
    "email": "", "telefon": "", "bank": "", "iban": "", "bic": ""
}


def connect(db_path: Path):
    """Verbindet sich mit der SQLite-Datenbank (erstellt sie, falls nicht vorhanden)."""
    conn = sqlite3.connect(str(db_path))
    conn.row_factory = sqlite3.Row
    return conn


def _column_exists(cur: sqlite3.Cursor, table: str, column: str) -> bool:
    info = cur.execute(f"PRAGMA table_info({table})").fetchall()
    return any(r[1] == column for r in info)


def init_db(conn: sqlite3.Connection):
    """Erstellt Tabellen, falls sie noch nicht existieren; migriert fehlende Spalten/Indizes."""
    cur = conn.cursor()
    cur.execute("PRAGMA foreign_keys = ON;")

    # ---------------- Stammdaten ----------------
    cur.execute("""
        CREATE TABLE IF NOT EXISTS firma (
            id INTEGER PRIMARY KEY CHECK (id = 1),
            name TEXT, strasse TEXT, plz TEXT, ort TEXT,
            email TEXT, telefon TEXT, bank TEXT, iban TEXT, bic TEXT
        );
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS artikel (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            nr TEXT, name TEXT, vk_preis REAL, einheit TEXT
        );
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS kunden (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT, strasse TEXT, plz TEXT, ort TEXT, email TEXT, telefon TEXT
        );
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS lieferanten (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT, strasse TEXT, plz TEXT, ort TEXT, email TEXT, telefon TEXT
        );
    """)

    # ---------------- Bewegungen ----------------
    cur.execute("""
        CREATE TABLE IF NOT EXISTS einkaeufe (
            uid INTEGER PRIMARY KEY,
            datum TEXT, artikel TEXT, menge REAL, einheit TEXT,
            preis_netto REAL, preis_brutto REAL, steuer INTEGER,
            lieferant TEXT, gesamt_brutto REAL, rechnungs_nr TEXT,
            status TEXT, bemerkung TEXT
        );
    """)

    # verkaeufe evtl. bereits ohne rechnung_nr vorhanden -> erst Tabelle, dann MIGRATION
    cur.execute("""
        CREATE TABLE IF NOT EXISTS verkaeufe (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            datum TEXT, artikel TEXT, menge REAL, einheit TEXT,
            preis REAL, steuer INTEGER, gesamt REAL, kunde TEXT
            -- rechnung_nr kommt per Migration dazu falls fehlt
        );
    """)
    # MIGRATION: rechnung_nr-Spalte nachrüsten, falls sie fehlt
    if not _column_exists(cur, "verkaeufe", "rechnung_nr"):
        cur.execute("ALTER TABLE verkaeufe ADD COLUMN rechnung_nr TEXT;")

    # Eindeutiger Index (NULLs erlaubt, echte Nummern einzigartig)
    cur.execute("CREATE UNIQUE INDEX IF NOT EXISTS idx_verkaeufe_rechnung_nr ON verkaeufe(rechnung_nr);")

    cur.execute("""
        CREATE TABLE IF NOT EXISTS bestand (
            artikel TEXT PRIMARY KEY,
            menge REAL, preis REAL, einheit TEXT
        );
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS privatentnahmen (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            datum TEXT, betrag REAL
        );
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS sperrliste (
            uid INTEGER PRIMARY KEY,
            status TEXT, entscheidung TEXT, bemerkung TEXT,
            datum TEXT, artikel TEXT, menge REAL, einheit TEXT,
            preis_netto REAL, preis_brutto REAL, steuer INTEGER,
            lieferant TEXT, rechnungs_nr TEXT
        );
    """)

    # ---------------- Settings ----------------
    cur.execute("""
        CREATE TABLE IF NOT EXISTS settings (
            key TEXT PRIMARY KEY,
            value TEXT
        );
    """)

    # Basisdaten absichern
    cur.execute("INSERT OR IGNORE INTO firma (id) VALUES (1);")
    cur.execute("INSERT OR IGNORE INTO settings(key, value) VALUES ('anfangsbestand', '0.0');")

    conn.commit()


def next_rechnung_nr(conn: sqlite3.Connection, prefix=None, width=5, yearly_reset=True) -> str:
    """
    Erzeugt eine neue, eindeutige Rechnungsnummer (transaktionssicher).
    - yearly_reset=True: Nummerierung pro Jahr; Schema: YYYY-00001
    - prefix: optionaler Präfix (z. B. 'BF') → 'BF-2025-00001'
    - width: Stellen der laufenden Nummer
    """
    cur = conn.cursor()
    # Sperrt schreibend – schützt vor Parallelvergabe
    cur.execute("BEGIN IMMEDIATE;")

    year = datetime.now().year
    if yearly_reset:
        row = cur.execute("SELECT value FROM settings WHERE key='rechnung_seq_year'").fetchone()
        cur_year = int(row[0]) if row else None
        if cur_year != year:
            seq = 1
            cur.execute("REPLACE INTO settings(key,value) VALUES('rechnung_seq_year', ?)", (str(year),))
        else:
            row = cur.execute("SELECT value FROM settings WHERE key='rechnung_seq'").fetchone()
            seq = int(row[0]) + 1 if row else 1
    else:
        row = cur.execute("SELECT value FROM settings WHERE key='rechnung_seq'").fetchone()
        seq = int(row[0]) + 1 if row else 1

    cur.execute("REPLACE INTO settings(key,value) VALUES('rechnung_seq', ?)", (str(seq),))
    conn.commit()

    core = f"{year}-{seq:0{width}d}" if yearly_reset else f"{seq:0{width}d}"
    return f"{prefix}-{core}" if prefix else core


def read_all(conn: sqlite3.Connection):
    """Liest alle Daten aus der Datenbank in ein Dict für die App."""
    cur = conn.cursor()

    # Firma
    row = cur.execute("SELECT * FROM firma WHERE id = 1").fetchone()
    firma = dict(row) if row else DEFAULT_FIRMA.copy()
    firma = {**DEFAULT_FIRMA, **{k: (firma.get(k) or '') for k in DEFAULT_FIRMA.keys()}}

    # Stammdaten
    artikel = [dict(r) for r in cur.execute("SELECT nr, name, vk_preis, einheit FROM artikel ORDER BY id").fetchall()]
    kunden = [dict(r) for r in cur.execute("SELECT name, strasse, plz, ort, email, telefon FROM kunden ORDER BY id").fetchall()]
    lieferanten = [dict(r) for r in cur.execute("SELECT name, strasse, plz, ort, email, telefon FROM lieferanten ORDER BY id").fetchall()]

    # Bewegungen
    einkaeufe = [dict(r) for r in cur.execute("SELECT * FROM einkaeufe ORDER BY datum, uid").fetchall()]
    verkaeufe = [dict(r) for r in cur.execute("SELECT * FROM verkaeufe ORDER BY datum, id").fetchall()]
    bestand = [dict(r) for r in cur.execute("SELECT * FROM bestand ORDER BY artikel").fetchall()]
    privatentnahmen = [dict(r) for r in cur.execute("SELECT * FROM privatentnahmen ORDER BY datum, id").fetchall()]
    sperrliste = [dict(r) for r in cur.execute("SELECT * FROM sperrliste ORDER BY datum, uid").fetchall()]

    # Settings
    ab_row = cur.execute("SELECT value FROM settings WHERE key='anfangsbestand'").fetchone()
    anfangsbestand = float(ab_row[0]) if ab_row else 0.0

    return {
        "stammdaten": {
            "firma": firma,
            "artikel": artikel,
            "kunden": kunden,
            "lieferanten": lieferanten,
        },
        "einkäufe": einkaeufe,
        "verkäufe": verkaeufe,
        "bestand": bestand,
        "anfangsbestand": anfangsbestand,
        "privatentnahmen": privatentnahmen,
        "sperrliste": sperrliste,
    }


def write_all(conn: sqlite3.Connection, data: dict):
    """Schreibt die aktuellen App-Daten vollständig in die DB (einfach & robust)."""
    cur = conn.cursor()
    cur.execute("BEGIN TRANSACTION;")

    # Firma
    f = data.get("stammdaten", {}).get("firma", {})
    cur.execute("""
        UPDATE firma SET name=?, strasse=?, plz=?, ort=?, email=?, telefon=?, bank=?, iban=?, bic=? WHERE id=1
    """, (
        f.get("name", ""), f.get("strasse", ""), f.get("plz", ""), f.get("ort", ""),
        f.get("email", ""), f.get("telefon", ""), f.get("bank", ""), f.get("iban", ""), f.get("bic", "")
    ))

    # Settings
    cur.execute("REPLACE INTO settings(key, value) VALUES('anfangsbestand', ?)",
                (str(float(data.get("anfangsbestand", 0.0))),))

    # Helper zum Neuschreiben von Tabellen
    def replace_table(table, cols, rows):
        cur.execute(f"DELETE FROM {table};")
        if rows:
            placeholders = ",".join(["?"] * len(cols))
            cur.executemany(
                f"INSERT INTO {table} ({','.join(cols)}) VALUES ({placeholders})",
                [[row.get(c) for c in cols] for row in rows]
            )

    replace_table("artikel", ["nr", "name", "vk_preis", "einheit"],
                  data.get("stammdaten", {}).get("artikel", []))
    replace_table("kunden", ["name", "strasse", "plz", "ort", "email", "telefon"],
                  data.get("stammdaten", {}).get("kunden", []))
    replace_table("lieferanten", ["name", "strasse", "plz", "ort", "email", "telefon"],
                  data.get("stammdaten", {}).get("lieferanten", []))

    replace_table("einkaeufe", ["uid", "datum", "artikel", "menge", "einheit", "preis_netto",
                                "preis_brutto", "steuer", "lieferant", "gesamt_brutto",
                                "rechnungs_nr", "status", "bemerkung"],
                  data.get("einkäufe", []))
    replace_table("verkaeufe", ["datum", "artikel", "menge", "einheit", "preis", "steuer", "gesamt", "kunde", "rechnung_nr"],
                  data.get("verkäufe", []))
    replace_table("bestand", ["artikel", "menge", "preis", "einheit"],
                  data.get("bestand", []))
    replace_table("privatentnahmen", ["datum", "betrag"],
                  data.get("privatentnahmen", []))
    replace_table("sperrliste", ["uid", "status", "entscheidung", "bemerkung", "datum", "artikel",
                                 "menge", "einheit", "preis_netto", "preis_brutto", "steuer",
                                 "lieferant", "rechnungs_nr"],
                  data.get("sperrliste", []))

    conn.commit()