#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
utils.py
Hilfsfunktionen für FinanzPlaner.
- Datumshilfen
- Euroformatierung
- Farbpaletten
- Geometrie/Math
- Tooltip-Unterstützung
"""

from __future__ import annotations
from datetime import datetime, date
import math


# ───────────────────────────────────────────────────────────────
#   Datum & Format
# ───────────────────────────────────────────────────────────────

def parse_date(s: str) -> datetime | None:
    """
    Versucht verschiedene Datumsformate zu parsen.
    """
    if not s:
        return None
    for fmt in ("%d.%m.%Y", "%Y-%m-%d", "%Y/%m/%d"):
        try:
            return datetime.strptime(s, fmt)
        except ValueError:
            pass
    return None


def format_euro(value: float) -> str:
    """
    Formatiert eine Zahl als Euro mit deutschem Dezimalformat.
    """
    s = f"{value:,.2f}"
    s = s.replace(",", "X").replace(".", ",").replace("X", ".")
    return s + " €"


def current_month_key() -> str:
    """
    Liefert 'YYYY-MM' des aktuellen Monats.
    """
    today = date.today()
    return today.strftime("%Y-%m")


def month_key(dt: datetime) -> str:
    """
    Liefert 'YYYY-MM' aus einem datetime.
    """
    return dt.strftime("%Y-%m")


# ───────────────────────────────────────────────────────────────
#   Farben für Diagramme
# ───────────────────────────────────────────────────────────────

CATEGORY_COLORS = [
    "#4c8bf5",  # Blau
    "#f55f4c",  # Rot
    "#4cf583",  # Grün
    "#f5c84c",  # Gelb
    "#b24cf5",  # Lila
    "#4cf5e9",  # Türkis
    "#f54ccd",  # Pink
    "#9ea2a7",  # Grau
]

BAR_INCOME_COLOR = "#4cf583"
BAR_EXPENSE_COLOR = "#f55f4c"

DOUGHNUT_HOVER_SCALE = 1.04
BAR_HOVER_SHADE = 0.8


def hex_to_rgb(hex_color: str) -> tuple[float, float, float]:
    """
    Hex in RGB (0..1)
    """
    hex_color = hex_color.lstrip("#")
    return (
        int(hex_color[0:2], 16) / 255,
        int(hex_color[2:4], 16) / 255,
        int(hex_color[4:6], 16) / 255,
    )


# ───────────────────────────────────────────────────────────────
#   Mathematische Hilfen
# ───────────────────────────────────────────────────────────────

def point_in_circle(px, py, cx, cy, r) -> bool:
    """
    Prüft, ob Punkt (px,py) in Kreis mit Mittelpunkt (cx,cy) und Radius r liegt.
    """
    return (px - cx) ** 2 + (py - cy) ** 2 <= r ** 2


def point_in_ring(px, py, cx, cy, r_outer, r_inner) -> bool:
    """
    Prüft, ob Punkt in Donut-Segment liegen könnte.
    """
    dist = math.sqrt((px - cx) ** 2 + (py - cy) ** 2)
    return r_inner <= dist <= r_outer


def angle_of_point(px, py, cx, cy) -> float:
    """
    Winkel (in Grad) eines Punktes relativ zum Kreismittelpunkt.
    0 Grad = rechts, 90 = oben (wegen GTK-Koordinaten umgekehrt).
    """
    dx = px - cx
    dy = py - cy
    angle = math.degrees(math.atan2(-dy, dx))  # y invertiert
    return (angle + 360) % 360


def shade_color(hex_color: str, factor: float) -> str:
    """
    Abdunkeln/Erhellen eines Hex-Farbwerts.
    factor < 1 → dunkler
    factor > 1 → heller
    """
    r, g, b = hex_to_rgb(hex_color)
    r = min(max(r * factor, 0), 1)
    g = min(max(g * factor, 0), 1)
    b = min(max(b * factor, 0), 1)
    return "#{:02x}{:02x}{:02x}".format(int(r * 255), int(g * 255), int(b * 255))


# ───────────────────────────────────────────────────────────────
#   Tooltip / Positionierung
# ───────────────────────────────────────────────────────────────

def clamp(value: float, minv: float, maxv: float) -> float:
    return max(minv, min(maxv, value))
