#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
charts.py

Interaktive Diagramme für FinanzPlaner:
- DonutChartWidget (Ausgaben nach Kategorien + Legende)
- BarChartWidget (Monatsbalken)
"""

from __future__ import annotations
import math
import cairo
import gi

gi.require_version("Gtk", "4.0")
from gi.repository import Gtk, Gdk

from utils import (
    hex_to_rgb,
    CATEGORY_COLORS,
    BAR_INCOME_COLOR,
    BAR_EXPENSE_COLOR,
    shade_color,
    point_in_ring,
    angle_of_point,
    format_euro,
)


# ╔══════════════════════════════════════════════════════════════╗
#   HELPER: Rounded Rectangle
# ╚══════════════════════════════════════════════════════════════╝

def draw_rounded_rect(ctx, x, y, w, h, r):
    """Zeichnet ein Rechteck mit abgerundeten Ecken."""
    degrees = math.pi / 180.0
    ctx.new_sub_path()
    ctx.arc(x + w - r, y + r, r, -90 * degrees, 0 * degrees)
    ctx.arc(x + w - r, y + h - r, r, 0 * degrees, 90 * degrees)
    ctx.arc(x + r, y + h - r, r, 90 * degrees, 180 * degrees)
    ctx.arc(x + r, y + r, r, 180 * degrees, 270 * degrees)
    ctx.close_path()


# ╔══════════════════════════════════════════════════════════════╗
#   DONUT-CHART (Kategorie-Verteilung)
# ╚══════════════════════════════════════════════════════════════╝

class DonutChartWidget(Gtk.DrawingArea):
    """
    Interaktives Donut-Diagramm mit Legende auf der rechten Seite.
    """

    def __init__(self, category_data: dict[str, float]):
        super().__init__()
        self.category_data = category_data
        self.sorted_items = list(category_data.items())
        self.hover_index = None
        self.on_segment_clicked = None  # callback(name)

        # Standardgröße
        self.set_content_width(450)
        self.set_content_height(250)

        # Events aktivieren
        motion = Gtk.EventControllerMotion()
        motion.connect("motion", self.on_motion)
        self.add_controller(motion)

        click = Gtk.GestureClick()
        click.connect("pressed", self.on_click)
        self.add_controller(click)

        self.set_draw_func(self.on_draw)

    def update_data(self, category_data: dict[str, float]):
        self.category_data = category_data
        # Sortieren nach Betrag absteigend für schönere Darstellung
        self.sorted_items = sorted(category_data.items(), key=lambda x: x[1], reverse=True)
        self.queue_draw()

    # ----------------------------------------------------------
    # GEOMETRIE-HELPER
    # ----------------------------------------------------------
    def _get_chart_geometry(self, width, height):
        """
        Berechnet Mittelpunkt und Radius des Donuts.
        Der Donut sitzt in der linken Hälfte (z.B. 55% der Breite).
        """
        chart_area_w = width * 0.55
        cx = chart_area_w / 2
        cy = height / 2
        
        # Radius basierend auf der kleineren Dimension
        r_outer = min(cx, cy) * 0.9
        r_inner = r_outer * 0.6
        
        return cx, cy, r_outer, r_inner

    # ----------------------------------------------------------
    # MOUSE EVENTS
    # ----------------------------------------------------------
    def on_motion(self, controller, x, y):
        w = self.get_allocated_width()
        h = self.get_allocated_height()
        
        cx, cy, r_outer, r_inner = self._get_chart_geometry(w, h)

        # 1. Prüfen, ob Maus im Ring ist
        if not point_in_ring(x, y, cx, cy, r_outer, r_inner):
            if self.hover_index is not None:
                self.hover_index = None
                self.queue_draw()
            return

        # 2. Winkel berechnen
        angle = angle_of_point(x, y, cx, cy)  # Grad
        total = sum(v for _, v in self.sorted_items) or 1
        start_angle = 0
        new_hover = None

        for i, (cat, val) in enumerate(self.sorted_items):
            span = 360 * (val / total)
            if start_angle <= angle < start_angle + span:
                new_hover = i
                break
            start_angle += span

        if new_hover != self.hover_index:
            self.hover_index = new_hover
            self.queue_draw()

    def on_click(self, gesture, n_press, x, y):
        if self.hover_index is None:
            # Klick irgendwo anders -> Filter reset
            if self.on_segment_clicked:
                self.on_segment_clicked(None)
            return

        cat, _ = self.sorted_items[self.hover_index]
        if self.on_segment_clicked:
            self.on_segment_clicked(cat)

    # ----------------------------------------------------------
    # DRAW
    # ----------------------------------------------------------
    def on_draw(self, area, ctx: cairo.Context, width, height):
        # Geometrie holen
        cx, cy, r_outer, r_inner = self._get_chart_geometry(width, height)
        total = sum(v for _, v in self.sorted_items) or 1

        # ---------------------------
        # 1. DONUT ZEICHNEN
        # ---------------------------
        start_angle = -90  # Start oben
        
        # Falls keine Daten da sind, grauen Ring zeichnen
        if not self.sorted_items:
             ctx.set_source_rgb(0.3, 0.3, 0.3)
             ctx.arc(cx, cy, r_outer, 0, 2*math.pi)
             ctx.arc_negative(cx, cy, r_inner, 0, 2*math.pi)
             ctx.fill()
             return

        for i, (cat, val) in enumerate(self.sorted_items):
            span = 360 * (val / total)
            end_angle = start_angle + span

            color_hex = CATEGORY_COLORS[i % len(CATEGORY_COLORS)]
            
            # Highlight beim Hover
            if i == self.hover_index:
                color_hex = shade_color(color_hex, 1.25)
            
            r, g, b = hex_to_rgb(color_hex)
            ctx.set_source_rgb(r, g, b)

            ctx.move_to(cx, cy)
            ctx.arc(cx, cy, r_outer, math.radians(start_angle), math.radians(end_angle))
            ctx.arc_negative(cx, cy, r_inner, math.radians(end_angle), math.radians(start_angle))
            ctx.close_path()
            ctx.fill()

            start_angle = end_angle

        # ---------------------------
        # 2. LEGENDE ZEICHNEN (Rechts)
        # ---------------------------
        self._draw_legend(ctx, width, height, total)
        
        # Tooltip nur zeichnen, wenn Hover aktiv (optional)
        if self.hover_index is not None:
             self._draw_tooltip(ctx, width, height)

    def _draw_legend(self, ctx, width, height, total):
        """Zeichnet eine Liste der Kategorien auf die rechte Seite."""
        left_margin = width * 0.60  # Startpunkt X für Legende
        top_margin = 20
        line_height = 24
        
        # Schriftart für Legende
        ctx.select_font_face("Sans", cairo.FONT_SLANT_NORMAL, cairo.FONT_WEIGHT_NORMAL)
        ctx.set_font_size(13)

        for i, (cat, val) in enumerate(self.sorted_items):
            # Nicht zu viele zeichnen, wenn Platz knapp
            y = top_margin + i * line_height
            if y > height - 20: 
                break 

            color_hex = CATEGORY_COLORS[i % len(CATEGORY_COLORS)]
            
            # Hover-Effekt auch in der Legende (Fett drucken)
            if i == self.hover_index:
                 ctx.select_font_face("Sans", cairo.FONT_SLANT_NORMAL, cairo.FONT_WEIGHT_BOLD)
            else:
                 ctx.select_font_face("Sans", cairo.FONT_SLANT_NORMAL, cairo.FONT_WEIGHT_NORMAL)

            # 1. Farb-Quadrat
            r, g, b = hex_to_rgb(color_hex)
            ctx.set_source_rgb(r, g, b)
            draw_rounded_rect(ctx, left_margin, y, 12, 12, 3)
            ctx.fill()

            # 2. Text (Kategorie)
            ctx.set_source_rgb(0.9, 0.9, 0.9) # Helle Schrift (ideal für Darkmode)
            ctx.move_to(left_margin + 20, y + 11)
            
            label_text = f"{cat}"
            ctx.show_text(label_text)

            # 3. Betrag (dahinter)
            extents = ctx.text_extents(label_text)
            x_amount = left_margin + 20 + extents.width + 10
            
            ctx.set_source_rgb(0.7, 0.7, 0.7) 
            ctx.move_to(x_amount, y + 11)
            ctx.show_text(format_euro(val))

    def _draw_tooltip(self, ctx, width, height):
        cat, val = self.sorted_items[self.hover_index]
        total = sum(v for _, v in self.sorted_items) or 1
        percent = (val / total) * 100

        text = f"{cat}\n{format_euro(val)} ({percent:.1f}%)"

        padding = 8
        lines = text.split("\n")
        ctx.set_font_size(14)

        # Textgröße bestimmen
        tw = max(ctx.text_extents(t)[2] for t in lines)
        th = len(lines) * 18

        box_w = tw + padding * 2
        box_h = th + padding * 2

        x = width - box_w - 10
        y = 10

        # Box hintergrund
        ctx.set_source_rgba(0, 0, 0, 0.7)
        draw_rounded_rect(ctx, x, y, box_w, box_h, 8)
        ctx.fill()

        # Text
        ctx.set_source_rgb(1, 1, 1)
        tx = x + padding
        ty = y + padding + 14
        for line in lines:
            ctx.move_to(tx, ty)
            ctx.show_text(line)
            ty += 18


# ╔══════════════════════════════════════════════════════════════╗
#   BAR-CHART (Monatsbalken)
# ╚══════════════════════════════════════════════════════════════╝

class BarChartWidget(Gtk.DrawingArea):
    """
    Interaktives Balkendiagramm:
    - Balken-Hover
    - Tooltip
    - Klick → callback(year, month)
    """

    def __init__(self, month_data: dict[str, float]):
        super().__init__()

        self.month_data = month_data
        self.sorted_keys = sorted(month_data.keys())
        self.hover_index = None
        self.on_bar_clicked = None  # callback(year, month)

        self.set_content_width(500)
        self.set_content_height(200)

        # Events
        motion = Gtk.EventControllerMotion()
        motion.connect("motion", self.on_motion)
        self.add_controller(motion)

        click = Gtk.GestureClick()
        click.connect("pressed", self.on_click)
        self.add_controller(click)

        self.set_draw_func(self.on_draw)

    def update_data(self, month_data: dict[str, float]):
        self.month_data = month_data
        self.sorted_keys = sorted(month_data.keys())
        self.queue_draw()

    # ----------------------------------------------------------
    # MOUSE EVENTS
    # ----------------------------------------------------------
    def on_motion(self, controller, x, y):
        n = len(self.sorted_keys)
        if n == 0:
            return

        width = self.get_allocated_width()
        height = self.get_allocated_height()

        bar_width = width / max(n, 1)

        new_index = int(x // bar_width)
        if new_index < 0 or new_index >= n:
            new_index = None

        if new_index != self.hover_index:
            self.hover_index = new_index
            self.queue_draw()

    def on_click(self, gesture, n_press, x, y):
        if self.hover_index is None:
            if self.on_bar_clicked:
                self.on_bar_clicked(None, None)
            return

        key = self.sorted_keys[self.hover_index]  # "2024-02"
        year = int(key.split("-")[0])
        month = int(key.split("-")[1])

        if self.on_bar_clicked:
            self.on_bar_clicked(year, month)

    # ----------------------------------------------------------
    # DRAWING
    # ----------------------------------------------------------
    def on_draw(self, area, ctx: cairo.Context, width, height):
        if not self.sorted_keys:
            return

        values = [self.month_data[k] for k in self.sorted_keys]
        max_val = max(values) or 1

        n = len(values)
        bar_width = width / n

        for i, key in enumerate(self.sorted_keys):
            val = values[i]
            bar_height = (val / max_val) * (height * 0.8)
            x = i * bar_width
            y = height - bar_height

            color = BAR_EXPENSE_COLOR
            if i == self.hover_index:
                color = shade_color(color, 1.2)

            r, g, b = hex_to_rgb(color)
            ctx.set_source_rgb(r, g, b)
            ctx.rectangle(x + 4, y, bar_width - 8, bar_height)
            ctx.fill()

        # Tooltip
        if self.hover_index is not None:
            self._draw_tooltip(ctx, width, height)

    # ----------------------------------------------------------
    # Tooltip
    # ----------------------------------------------------------
    def _draw_tooltip(self, ctx, width, height):
        key = self.sorted_keys[self.hover_index]  # "2024-02"
        val = self.month_data[key]

        text = f"{key}\n{format_euro(val)}"

        padding = 8
        lines = text.split("\n")
        ctx.set_font_size(14)

        tw = max(ctx.text_extents(t)[2] for t in lines)
        th = len(lines) * 18

        box_w = tw + padding * 2
        box_h = th + padding * 2

        # rechts oben
        x = width - box_w - 10
        y = 10

        ctx.set_source_rgba(0, 0, 0, 0.7)
        draw_rounded_rect(ctx, x, y, box_w, box_h, 8)
        ctx.fill()

        ctx.set_source_rgb(1, 1, 1)
        tx = x + padding
        ty = y + padding + 14

        for line in lines:
            ctx.move_to(tx, ty)
            ctx.show_text(line)
            ty += 18